/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/java_utility.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/utf8.h"

#include <stdlib.h>

#include <functional>
#include <iostream>

#ifdef __ANDROID__
#define ENABLE_PTHREAD_DETACH_OPTIMIZATION
#endif

#ifdef ENABLE_PTHREAD_DETACH_OPTIMIZATION
#include <pthread.h>
#endif

namespace {
const jint kMinLocalReferenceTableCapacity = 64;

#ifdef ENABLE_PTHREAD_DETACH_OPTIMIZATION

/**
 * The key is used to store the JNIEnv value associated with a native thread when it is attached to the JVM.
 */
pthread_key_t gPThreadKey = 0;

void RegisterThreadCleanupCallback(JNIEnv* jEnv) {
  TTV_ASSERT(gPThreadKey != 0);

  // Make sure the value isn't already set
  void* storedJEnv = pthread_getspecific(gPThreadKey);

  if (storedJEnv == nullptr) {
    // Associate the JNIEnv value with the thread
    int ret = pthread_setspecific(gPThreadKey, jEnv);
    TTV_ASSERT(ret == 0);
  }
}

#endif
}  // namespace

// These are currently accessed from outside the module as well and are thus not static
JavaVM* ttv::binding::java::gGlobalJavaVirtualMachine = nullptr;
JNIEnv* ttv::binding::java::gActiveJavaEnvironment = nullptr;

int ttv::binding::java::ScopedJavaEnvironmentCacher::mCacheCount = 0;

void ttv::binding::java::JniThreadInitialize() {
#ifdef ENABLE_PTHREAD_DETACH_OPTIMIZATION
  if (gPThreadKey == 0) {
    // Setup a cleanup callback for threads as they are destroyed.  This
    // will be used to detach them from the JVM if needed.
    int ret = pthread_key_create(&gPThreadKey, [](void* threadJEnv) {
      auto jvm = ttv::binding::java::gGlobalJavaVirtualMachine;

      if (threadJEnv != nullptr && jvm != nullptr) {
        jvm->DetachCurrentThread();
      }
    });

    TTV_ASSERT(ret == 0);
  }
#endif
}

ttv::binding::java::JavaLocalReferenceDeleter::JavaLocalReferenceDeleter(
  JNIEnv* jEnv, jobject jObject, const char* name)
    : mjEnv(jEnv), mObject(jObject), mName(name) {}

ttv::binding::java::JavaLocalReferenceDeleter::~JavaLocalReferenceDeleter() {
  if (mjEnv != nullptr && mObject != nullptr) {
    mjEnv->DeleteLocalRef(mObject);
  }
}

ttv::binding::java::ScopedJavaUTFStringConverter::ScopedJavaUTFStringConverter(JNIEnv* jEnv, jstring jstr) {
  mjEnv = jEnv;
  mJavaString = jstr;

  if (jstr != nullptr) {
    JavaClassInfo& stringInfo = GetJavaClassInfo_String(jEnv);
    mNativeString = mjEnv->GetStringUTFChars(jstr, nullptr);
    mCharacterLength = static_cast<int>(mjEnv->CallIntMethod(jstr, stringInfo.methods["length"]));
    mByteLength = ttv::CountUtf8Bytes(mNativeString, mCharacterLength);
  } else {
    mNativeString = nullptr;
    mByteLength = 0;
    mCharacterLength = 0;
  }
}

ttv::binding::java::ScopedJavaUTFStringConverter::~ScopedJavaUTFStringConverter() {
  if (mJavaString) {
    mjEnv->ReleaseStringUTFChars(mJavaString, mNativeString);
    mNativeString = nullptr;
  }
}

const char* ttv::binding::java::ScopedJavaUTFStringConverter::GetNativeString() {
  return mNativeString;
}

ttv::binding::java::ScopedJavaWcharStringConverter::ScopedJavaWcharStringConverter(JNIEnv* jEnv, jstring jstr) {
  mjEnv = jEnv;
  mJavaString = jstr;

  if (jstr != nullptr) {
    const char* utf = mjEnv->GetStringUTFChars(jstr, nullptr);

    if (utf != nullptr) {
      std::string str(utf);
      mSTDWideString = std::wstring(str.begin(), str.end());
      mNativeString = mSTDWideString.c_str();

      mjEnv->ReleaseStringUTFChars(mJavaString, utf);
    }
  } else {
    mNativeString = nullptr;
  }
}

const wchar_t* ttv::binding::java::ScopedJavaWcharStringConverter::GetNativeString() const {
  return mNativeString;
}

ttv::binding::java::ScopedJavaEnvironmentCacher::ScopedJavaEnvironmentCacher(JNIEnv* jenv) {
  TTV_ASSERT(jenv != nullptr);

  if (mCacheCount == 0) {
    gActiveJavaEnvironment = jenv;

    // Ensure we have a large enough local reference table
    jint result = jenv->EnsureLocalCapacity(kMinLocalReferenceTableCapacity);
    if (result != 0) {
      ttv::trace::Message("jni", MessageLevel::Error, "Call to set EnsureLocalCapacity to %d failed",
        static_cast<int>(kMinLocalReferenceTableCapacity));
    }
  } else {
    TTV_ASSERT(gActiveJavaEnvironment == jenv);
    if (gActiveJavaEnvironment != jenv) {
      ttv::trace::Message("jni", MessageLevel::Error, "Scoped JNI changed before releasing");
    }
  }

  mCacheCount++;
}

ttv::binding::java::ScopedJavaEnvironmentCacher::~ScopedJavaEnvironmentCacher() {
  TTV_ASSERT(mCacheCount > 0);
  mCacheCount--;

  if (mCacheCount == 0) {
    gActiveJavaEnvironment = nullptr;
  }
}

int ttv::binding::java::ScopedJavaEnvironmentCacher::GetMinLocalReferenceTableCapacity() {
  return kMinLocalReferenceTableCapacity;
}

ttv::binding::java::AutoJEnv::AutoJEnv() : mJvm(gGlobalJavaVirtualMachine), mJEnv(nullptr), mNeedsDetach(false) {
  TTV_ASSERT(mJvm != nullptr);

  Lock();
}

ttv::binding::java::AutoJEnv::AutoJEnv(JavaVM* jvm) : mJvm(jvm), mJEnv(nullptr) {
  TTV_ASSERT(mJvm != nullptr);

  Lock();
}

ttv::binding::java::AutoJEnv::~AutoJEnv() {
  Unlock();
}

JNIEnv* ttv::binding::java::AutoJEnv::operator->() {
  return mJEnv;
}

ttv::binding::java::AutoJEnv::operator JNIEnv*() {
  return mJEnv;
}

bool ttv::binding::java::AutoJEnv::Lock() {
  void* jni = nullptr;
  jint jAttachRet = JNI_OK;
  jint jEnvStat = mJvm->GetEnv(&jni, JNI_VERSION_1_6);

  if (jEnvStat == JNI_EDETACHED) {
    // ttv::trace::Message("bindings", MessageLevel::Debug, "AttachCurrentThread");

    TTV_ASSERT(jni == nullptr);

    // This adapter is needed since the AttachCurrentThread API is slightly different between regular Java and the
    // Android NDK
    class JniParameterAdapter {
     public:
      JniParameterAdapter(void** jniArg) : mJni(jniArg) {}
      operator void**() { return mJni; }
      operator JNIEnv**() { return reinterpret_cast<JNIEnv**>(mJni); }

     private:
      void** mJni;
    };

    JniParameterAdapter adapter(&jni);
    jAttachRet = mJvm->AttachCurrentThread(adapter, nullptr);
    TTV_ASSERT(jAttachRet == JNI_OK);

    if (jAttachRet == JNI_OK) {
      TTV_ASSERT(jni != nullptr);

#ifdef ENABLE_PTHREAD_DETACH_OPTIMIZATION
      RegisterThreadCleanupCallback(static_cast<JNIEnv*>(jni));

      mNeedsDetach = false;
#else
      mNeedsDetach = true;
#endif
    } else {
      ttv::trace::Message("bindings", MessageLevel::Error, "AutoJEnv::Lock: AttachCurrentThread failed");
    }
  }
  // Already attached
  else if (jEnvStat == JNI_OK) {
    TTV_ASSERT(jni != nullptr);
  }
  // Failed
  else if (jEnvStat == JNI_EVERSION) {
    ttv::trace::Message(
      "bindings", MessageLevel::Error, "JavaVMReference::LockInternal: JNI_VERSION_1_6 not supported");
    return false;
  }

  mJEnv = static_cast<JNIEnv*>(jni);

  return true;
}

void ttv::binding::java::AutoJEnv::Unlock() {
// Allow the thread cleanup callback to detach
#ifndef ENABLE_PTHREAD_DETACH_OPTIMIZATION
  if (mJEnv == nullptr) {
    return;
  }

  if (mNeedsDetach) {
    // ttv::trace::Message("bindings", MessageLevel::Debug, "DetachCurrentThread");
    jint jDetachRet = mJvm->DetachCurrentThread();
    TTV_ASSERT(jDetachRet == JNI_OK);

    if (jDetachRet != JNI_OK) {
      ttv::trace::Message("bindings", MessageLevel::Error, "AutoJEnv::Unlock: DetachCurrentThread failed");
    }
  }

  mJEnv = nullptr;
  mNeedsDetach = false;
#endif
}

ttv::binding::java::GlobalJavaObjectReference::GlobalJavaObjectReference() : mObject(nullptr) {}

ttv::binding::java::GlobalJavaObjectReference::~GlobalJavaObjectReference() {
  Release();
}

bool ttv::binding::java::GlobalJavaObjectReference::Bind(JNIEnv* jEnv, jobject jObject) {
  Release();

  if (jObject == nullptr) {
    return false;
  }

  // Cache a global reference
  mObject = jEnv->NewGlobalRef(jObject);
  if (mObject == nullptr) {
    ttv::trace::Message(
      "bindings", MessageLevel::Error, "GlobalJavaObjectReference::Bind: Could not cache global reference");
    Release();
    return false;
  }

  return true;
}

void ttv::binding::java::GlobalJavaObjectReference::Release(JNIEnv* jEnv) {
  if (mObject != nullptr) {
    jEnv->DeleteGlobalRef(mObject);
    mObject = nullptr;
  }
}

void ttv::binding::java::GlobalJavaObjectReference::Release() {
  if (mObject != nullptr) {
    TTV_ASSERT(gGlobalJavaVirtualMachine);
    AutoJEnv jEnv(gGlobalJavaVirtualMachine);
    Release(jEnv);
  }
}

bool ttv::binding::java::CacheJavaVirtualMachine(JNIEnv* jEnv) {
  if (gGlobalJavaVirtualMachine == nullptr) {
    jint ret = jEnv->GetJavaVM(&gGlobalJavaVirtualMachine);
    TTV_ASSERT(gGlobalJavaVirtualMachine != nullptr);
    TTV_ASSERT(ret == JNI_OK && gGlobalJavaVirtualMachine != nullptr);
    if (ret != JNI_OK || gGlobalJavaVirtualMachine == nullptr) {
      ttv::trace::Message("bindings", MessageLevel::Error, "Could not get JavaVM");
    }
  }

  return gGlobalJavaVirtualMachine != nullptr;
}

void ttv::binding::java::LoadAllUtilityJavaClassInfo(JNIEnv* jEnv) {
  GetJavaClassInfo_Boolean(jEnv);
  GetJavaClassInfo_Integer(jEnv);
  GetJavaClassInfo_Long(jEnv);
  GetJavaClassInfo_Float(jEnv);
  GetJavaClassInfo_String(jEnv);
  GetJavaClassInfo_Charset(jEnv);
  GetJavaClassInfo_HashSet(jEnv);
  GetJavaClassInfo_HashMap(jEnv);

  GetJavaClassInfo_EnumValue(jEnv);
  GetJavaClassInfo_ErrorCode(jEnv);
  GetJavaClassInfo_ErrorResult(jEnv);
  GetJavaClassInfo_HttpParameter(jEnv);
  GetJavaClassInfo_HttpRequestResult(jEnv);
  GetJavaClassInfo_IHttpRequestProvider(jEnv);
  GetJavaClassInfo_ISocket(jEnv);
  GetJavaClassInfo_ISocketFactory(jEnv);
  GetJavaClassInfo_IWebSocket(jEnv);
  GetJavaClassInfo_IWebSocketFactory(jEnv);
  GetJavaClassInfo_MessageLevel(jEnv);
  GetJavaClassInfo_NativeProxy(jEnv);
  GetJavaClassInfo_Result(jEnv);
  GetJavaClassInfo_ResultContainer(jEnv);
  GetJavaClassInfo_SuccessResult(jEnv);
  GetJavaClassInfo_TaskFunction(jEnv);
  GetJavaClassInfo_TaskId(jEnv);
  GetJavaClassInfo_TaskParams(jEnv);
  GetJavaClassInfo_WebSocketMessageType(jEnv);
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Boolean(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/Boolean");

    LookupJavaMethod(jEnv, info, "<init>", "(Z)V");  // ctor that takes a jboolean
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Integer(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/Integer");

    LookupJavaMethod(jEnv, info, "<init>", "(I)V");  // ctor that takes a jint
    LookupJavaMethod(jEnv, info, "intValue", "()I");
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Long(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/Long");

    LookupJavaMethod(jEnv, info, "<init>", "(J)V");    // public java.lang.Long(long);
    LookupJavaMethod(jEnv, info, "longValue", "()J");  // public long longValue();
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Float(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/Float");

    LookupJavaMethod(jEnv, info, "<init>", "(F)V");  // public java.lang.Float(float);
    LookupJavaMethod(jEnv, info, "<init>", "(D)V");  // public java.lang.Float(double);

    LookupJavaMethod(jEnv, info, "floatValue", "()F");  // public float floatValue();
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Double(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/Double");

    LookupJavaMethod(jEnv, info, "<init>", "(D)V");  // public java.lang.Double(double);

    LookupJavaMethod(jEnv, info, "doubleValue", "()D");  // public double doubleValue();
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_String(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/lang/String");

    LookupJavaMethod(jEnv, info, "<init>", "([BLjava/nio/charset/Charset;)V");

    LookupJavaMethod(jEnv, info, "length", "()I");
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_Charset(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/nio/charset/Charset");

    LookupJavaStaticMethod(jEnv, info, "forName", "(Ljava/lang/String;)Ljava/nio/charset/Charset;");
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_HashSet(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/util/HashSet");

    LookupJavaMethod(jEnv, info, "<init>", "()V");

    LookupJavaMethod(jEnv, info, "add", "(Ljava/lang/Object;)Z");
    LookupJavaMethod(jEnv, info, "remove", "(Ljava/lang/Object;)Z");
    LookupJavaMethod(jEnv, info, "clear", "()V");
    LookupJavaMethod(jEnv, info, "size", "()I");
  }

  return info;
}

ttv::binding::java::JavaClassInfo& ttv::binding::java::GetJavaClassInfo_HashMap(JNIEnv* jEnv) {
  static JavaClassInfo info;
  static bool initialized = false;
  if (!initialized) {
    initialized = true;

    LookupJavaClass(jEnv, info, "java/util/HashMap");

    LookupJavaMethod(jEnv, info, "<init>", "()V");

    LookupJavaMethod(jEnv, info, "isEmpty", "()Z");
    LookupJavaMethod(jEnv, info, "get", "(Ljava/lang/Object;)Ljava/lang/Object;");
    LookupJavaMethod(jEnv, info, "put", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
  }

  return info;
}

jstring ttv::binding::java::GetJavaInstance_StringWithEncoding(JNIEnv* jEnv, const std::string& str) {
  jstring jString = nullptr;
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEncoding, jEnv->NewStringUTF("UTF-8"));

  if (jEncoding == nullptr) {
    ttv::trace::Message("jni", MessageLevel::Error, "GetJavaInstance_StringWithEncoding: Failed to allocate string");
  } else {
    // Get the Charset instance for the String ctor
    JavaClassInfo& charsetInfo = GetJavaClassInfo_Charset(jEnv);
    static const auto forNameMethod = charsetInfo.staticMethods["forName"];
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jCharset,
      jEnv->CallStaticObjectMethod(charsetInfo.klass, forNameMethod, static_cast<jobject>(jEncoding)));
    if (jCharset != nullptr) {
      AUTO_DELETE_LOCAL_REF(jEnv, jbyteArray, jBytes, jEnv->NewByteArray(static_cast<jsize>(str.size())));
      jEnv->SetByteArrayRegion(jBytes, 0, static_cast<jsize>(str.size()), reinterpret_cast<const jbyte*>(str.data()));

      // Call the String ctor with the raw bytes
      JavaClassInfo& stringInfo = GetJavaClassInfo_String(jEnv);
      static const auto initMethod = stringInfo.methods["<init>"];
      jobject jConverted = jEnv->NewObject(stringInfo.klass, initMethod, jBytes, jCharset);
      jString = static_cast<jstring>(jConverted);

      // Unhandled input so just return a null string
      if (jString == nullptr) {
        ttv::trace::Message("jni", MessageLevel::Error,
          "GetJavaInstance_StringWithEncoding: Failed to create string with encoding: UTF-8");

        jEnv->ExceptionClear();
      }
    } else {
      ttv::trace::Message(
        "jni", MessageLevel::Error, "GetJavaInstance_StringWithEncoding: Failed to get Charset for: UTF-8");
    }
  }

  return jString;
}

jobject ttv::binding::java::GetJavaInstance_Boolean(JNIEnv* jEnv, bool value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Boolean(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jboolean>(value));
}

jobject ttv::binding::java::GetJavaInstance_Integer(JNIEnv* jEnv, int32_t value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Integer(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jint>(value));
}

jobject ttv::binding::java::GetJavaInstance_Integer(JNIEnv* jEnv, uint32_t value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Integer(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jint>(value));
}

jobject ttv::binding::java::GetJavaInstance_Long(JNIEnv* jEnv, uint64_t value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Long(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jlong>(value));
}

jobject ttv::binding::java::GetJavaInstance_Long(JNIEnv* jEnv, int64_t value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Long(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jlong>(value));
}

jobject ttv::binding::java::GetJavaInstance_Float(JNIEnv* jEnv, float value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Float(jEnv);

  // We cast to a double here, because double is the only floating-point type that is
  // accepted by C vararg lists.
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jdouble>(value));
}

jobject ttv::binding::java::GetJavaInstance_Double(JNIEnv* jEnv, double value) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_Double(jEnv);
  return jEnv->NewObject(info.klass, info.methods["<init>"], static_cast<jdouble>(value));
}

jstring ttv::binding::java::GetJavaInstance_String(JNIEnv* jEnv, const char* str) {
  if (str == nullptr) {
    return nullptr;
  } else {
    return GetJavaInstance_String(jEnv, std::string(str));
  }
}

jstring ttv::binding::java::GetJavaInstance_String(JNIEnv* jEnv, const std::string& str) {
  TTV_ASSERT(jEnv);
  bool hasFourByteChars = false;

  if (IsValidUtf8(str.c_str(), hasFourByteChars) && !hasFourByteChars) {
    return jEnv->NewStringUTF(str.c_str());
  } else {
    // If invalid or we have four byte characters then create string with java String ctor
    jstring jString = GetJavaInstance_StringWithEncoding(jEnv, str);
    if (jString != nullptr) {
      return jString;
    }
    jEnv->ExceptionClear();

    return nullptr;
  }
}

jobject ttv::binding::java::GetJavaInstance_EnumValue(JNIEnv* jEnv, const ttv::EnumValue& value) {
  JavaClassInfo& info = GetJavaClassInfo_EnumValue(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, value.name));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  jEnv->SetIntField(jInstance, info.fields["value"], static_cast<jint>(value.value));

  return jInstance;
}

jobjectArray ttv::binding::java::GetJavaInstance_EnumValueArray(JNIEnv* jEnv, const std::vector<ttv::EnumValue>& arr) {
  jobjectArray jArray = GetJavaInstance_Array(jEnv, GetJavaClassInfo_EnumValue(jEnv), static_cast<uint32_t>(arr.size()),
    [jEnv, &arr](uint32_t index) -> jobject { return GetJavaInstance_EnumValue(jEnv, arr[index]); });

  return jArray;
}

jobjectArray ttv::binding::java::GetJavaInstance_StringArray(JNIEnv* jEnv, const std::vector<std::string>& arr) {
  JavaClassInfo& stringInfo = GetJavaClassInfo_String(jEnv);

  jobjectArray jArray = jEnv->NewObjectArray(static_cast<jint>(arr.size()), stringInfo.klass, nullptr);

  for (size_t i = 0; i < arr.size(); ++i) {
    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jStr, GetJavaInstance_String(jEnv, arr[i]));

    jEnv->SetObjectArrayElement(jArray, static_cast<jsize>(i), jStr);
  }

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_StringHashMap(JNIEnv* jEnv, const std::map<std::string, std::string>& map) {
  JavaClassInfo& hashMapInfo = GetJavaClassInfo_HashMap(jEnv);

  jobject jHashMap = jEnv->NewObject(hashMapInfo.klass, hashMapInfo.methods["<init>"]);
  jmethodID addMethodId = hashMapInfo.methods["put"];

  for (auto kvp : map) {
    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jKey, GetJavaInstance_String(jEnv, kvp.first));
    AUTO_DELETE_LOCAL_REF(jEnv, jstring, jValue, GetJavaInstance_String(jEnv, kvp.second));

    jEnv->CallObjectMethod(jHashMap, addMethodId, jKey, jValue);
  }

  return jHashMap;
}

jobject ttv::binding::java::GetJavaInstance_ErrorCode(JNIEnv* jEnv, TTV_ErrorCode err) {
  TTV_ASSERT(jEnv);

  JavaClassInfo& info = GetJavaClassInfo_ErrorCode(jEnv);
  return jEnv->CallStaticObjectMethod(info.klass, info.staticMethods["lookupValue"], static_cast<jint>(err));
}

jobject ttv::binding::java::GetJavaInstance_ResultContainer(JNIEnv* jEnv) {
  JavaClassInfo& info = GetJavaClassInfo_ResultContainer(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_GetResultFromResultContainer(JNIEnv* jEnv, jobject jResultContainer) {
  JavaClassInfo& info = GetJavaClassInfo_ResultContainer(jEnv);

  return jEnv->GetObjectField(jResultContainer, info.fields["result"]);
}

jobject ttv::binding::java::GetJavaInstance_TaskId(JNIEnv* jEnv, TaskId taskId) {
  JavaClassInfo& info = GetJavaClassInfo_TaskId(jEnv);

  jobject jTaskId = jEnv->NewObject(info.klass, info.methods["<init>"]);
  jEnv->SetLongField(jTaskId, info.fields["id"], static_cast<jlong>(taskId));

  return jTaskId;
}

void ttv::binding::java::GetNativeInstance_StringVector(
  JNIEnv* jEnv, jobjectArray jArray, std::vector<std::string>& result) {
  if (jArray == nullptr) {
    return;
  }

  jsize size = jEnv->GetArrayLength(jArray);
  for (jsize i = 0; i < size; ++i) {
    jstring jStringValue = static_cast<jstring>(jEnv->GetObjectArrayElement(jArray, i));
    ScopedJavaUTFStringConverter stringValue(jEnv, jStringValue);
    result.emplace_back(stringValue.GetNativeString());
  }
}

void ttv::binding::java::SetResultContainerResult(JNIEnv* jEnv, jobject jResultContainer, jobject jResult) {
  JavaClassInfo& resultContainerInfo = GetJavaClassInfo_ResultContainer(jEnv);
  static const auto resultField = resultContainerInfo.fields["result"];

  jEnv->SetObjectField(jResultContainer, resultField, jResult);
}

jobjectArray ttv::binding::java::GetJavaInstance_Array(JNIEnv* jEnv, JavaClassInfo& javaArrayTypeClassInfo,
  const uint32_t size, std::function<jobject(uint32_t index)> entryFunc) {
  jobjectArray jArray = jEnv->NewObjectArray(static_cast<jsize>(size), javaArrayTypeClassInfo.klass, nullptr);

  for (uint32_t j = 0; j < size; ++j) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jEntry, entryFunc(j));
    jEnv->SetObjectArrayElement(jArray, static_cast<jsize>(j), jEntry);
  }

  return jArray;
}

void ttv::binding::java::GetNativeFromJava_ByteArray(JNIEnv* jEnv, jbyteArray jSource, std::vector<uint8_t>& dest) {
  if (jSource != nullptr) {
    jsize jLength = jEnv->GetArrayLength(jSource);
    dest.resize(static_cast<size_t>(jLength));
    jEnv->GetByteArrayRegion(jSource, 0, jLength, reinterpret_cast<jbyte*>(&dest[0]));
  } else {
    dest.resize(0);
  }
}

jobject ttv::binding::java::GetJavaInstance_SuccessResult(JNIEnv* jEnv, jobject jResultObject) {
  JavaClassInfo& successResultInfo = GetJavaClassInfo_SuccessResult(jEnv);
  jobject jSuccessResult = jEnv->NewObject(successResultInfo.klass, successResultInfo.methods["<init>"], jResultObject);

  return jSuccessResult;
}

jobject ttv::binding::java::GetJavaInstance_ErrorResult(JNIEnv* jEnv, TTV_ErrorCode ec) {
  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode, GetJavaInstance_ErrorCode(jEnv, ec));

  JavaClassInfo& errorResultInfo = GetJavaClassInfo_ErrorResult(jEnv);
  jobject jErrorResult = jEnv->NewObject(errorResultInfo.klass, errorResultInfo.methods["<init>"], jErrorCode);

  return jErrorResult;
}

jobject ttv::binding::java::GetJavaInstance_HttpParameter(JNIEnv* jEnv, const ttv::HttpParam& param) {
  JavaClassInfo& info = GetJavaClassInfo_HttpParameter(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, param.paramName));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jValue, GetJavaInstance_String(jEnv, param.paramValue));
  jEnv->SetObjectField(jInstance, info.fields["value"], jValue);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_HttpParameterArray(
  JNIEnv* jEnv, const std::vector<ttv::HttpParam>& params) {
  JavaClassInfo& info = GetJavaClassInfo_HttpParameter(jEnv);

  jobjectArray jArray = jEnv->NewObjectArray(static_cast<jsize>(params.size()), info.klass, nullptr);

  for (size_t i = 0; i < params.size(); ++i) {
    const ttv::HttpParam& param = params[i];

    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jInstance, GetJavaInstance_HttpParameter(jEnv, param));

    jEnv->SetObjectArrayElement(jArray, static_cast<jsize>(i), jInstance);
  }

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_HttpRequestResult(JNIEnv* jEnv) {
  JavaClassInfo& info = GetJavaClassInfo_HttpRequestResult(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  return jInstance;
}

void ttv::binding::java::GetNativeInstance_HttpRequestResult(JNIEnv* jEnv, jobject jRequestResult, uint& statusCode,
  std::map<std::string, std::string>& resultHeaders, std::vector<char>& response) {
  JavaClassInfo& httpRequestResultInfo = GetJavaClassInfo_HttpRequestResult(jEnv);

  statusCode = static_cast<uint>(jEnv->GetIntField(jRequestResult, httpRequestResultInfo.fields["statusCode"]));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jResponse,
    static_cast<jstring>(jEnv->GetObjectField(jRequestResult, httpRequestResultInfo.fields["response"])));
  if (jResponse != nullptr) {
    ScopedJavaUTFStringConverter responseConverter(jEnv, jResponse);

    response.resize(static_cast<size_t>(responseConverter.GetByteLength()));
    memcpy(
      response.data(), responseConverter.GetNativeString(), static_cast<size_t>(responseConverter.GetByteLength()));
  }

  JavaClassInfo& httpParameterInfo = GetJavaClassInfo_HttpParameter(jEnv);

  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jResponseHeadersArray,
    static_cast<jobjectArray>(jEnv->GetObjectField(jRequestResult, httpRequestResultInfo.fields["headers"])));
  if (jResponseHeadersArray != nullptr) {
    int length = static_cast<int>(jEnv->GetArrayLength(jResponseHeadersArray));

    for (int i = 0; i < length; ++i) {
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jParam, jEnv->GetObjectArrayElement(jResponseHeadersArray, i));

      if (jParam != nullptr) {
        AUTO_DELETE_LOCAL_REF(
          jEnv, jstring, jName, static_cast<jstring>(jEnv->GetObjectField(jParam, httpParameterInfo.fields["name"])));
        AUTO_DELETE_LOCAL_REF(
          jEnv, jstring, jValue, static_cast<jstring>(jEnv->GetObjectField(jParam, httpParameterInfo.fields["value"])));

        ScopedJavaUTFStringConverter nameConverter(jEnv, jName);
        ScopedJavaUTFStringConverter valueConverter(jEnv, jValue);

        const char* name = nameConverter.GetNativeString();
        const char* value = valueConverter.GetNativeString();

        if (name != nullptr && value != nullptr) {
          resultHeaders[name] = value;
        }
      }
    }
  }
}
