/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/httprequest.h"

namespace ttv {
class OrbisHttpRequest;
}

/**
 * Implementation of the HttpRequest primitive for PS4
 *
 * Built upon the PS4 http (<libhttp.h>) and net (<net.h>) libraries
 * See the PS4 dev reference: https://ps4.siedev.net/resources/documents/SDK/4.500/Http-Overview/__document_toc.html
 *
 * Two ways to initialize:
 *   1) Default constructor where PS4 library initialization functions will be called
 *   2) Pass in handles for PS4 library contexts that have been initialized elsewhere
 */
class ttv::OrbisHttpRequest : public HttpRequest {
 public:
  OrbisHttpRequest();

  /**
   * Initialize() needs to be called (globally) before SendHttpRequest() can be called
   *
   * It will set up all the underlying PS4 (Net, SSL, Http) libraries and the memory pools for them
   * mLibHttpCtxId will be set with the handle for the Http library to use for SendHttpRequest
   *
   * Returns:
   *   TTV_EC_HTTPREQUEST_ERROR if initialization failed
   *   TTV_EC_ALREADY_INITIALIZED if we have previously already initialized
   *   TTV_EC_SUCCESS if initialization succeeded
   */
  TTV_ErrorCode Initialize();

  /**
   * The overloaded version of Initialize() can be used to used to set the member handle values
   * if we've initialized the underlying PS4 libraries elsewhere.
   *
   * Returns TTV_EC_SUCCESS (assumes that the handles passed are valid and can be used)
   */
  TTV_ErrorCode Initialize(int libNetMemId, int libSslCtxId, int libHttpCtxId);

  /**
   * Shutdown() needs to be called (globally) one we're done with OrbisHttpRequest (and sending requests)
   *
   * Will shutdown the underlying PS4 libraries and free the memory pools associated with them
   * Only call this once all the http requests have been finished
   *
   * Returns:
   *   TTV_EC_SUCCESS if shutdown succeeded
   *   TTV_EC_NOT_INITIALIZED if we haven't initialized already and don't need shutdown
   */
  TTV_ErrorCode Shutdown();

  /**
   * PS4 implementation of HttpRequest::SendHttpRequest
   * Uses PS4 libraries to send a http request and calls headersCallback/responseCallback with header/response values.
   *
   * Returns:
   *   TTV_EC_SUCCESS if the http request is successfully sent
   *   TTV_EC_NOT_INITIALIZED if the underlying PS4 libraries haven't been initialized (with Initialize())
   *   TTV_EC_HTTPREQUEST_ERROR if the http request failed for any reason
   */
  virtual TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
    const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
    HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
    HttpRequestCallback responseCallback, void* userData);

 private:
  int mLibNetMemId;
  int mLibSslCtxId;
  int mLibHttpCtxId;
  bool mInitialized;
};
