/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/orbishttprequest.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/stringutilities.h"

#include <libhttp.h>
#include <libssl.h>
#include <net.h>

namespace {
const int kNetHeapSize = 16 * 1024;
const int kSslHeapSize = 304 * 1024;
const int kHttpHeapSize = 512 * 1024;
const char* kPs4UserAgent = "PS4/1.00";
}  // namespace

ttv::OrbisHttpRequest::OrbisHttpRequest()
    : mLibNetMemId(-1), mLibSslCtxId(-1), mLibHttpCtxId(-1), mInitialized(false) {}

TTV_ErrorCode ttv::OrbisHttpRequest::Initialize() {
  if (mInitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  if (sceNetInit() < 0) {
    ec = TTV_EC_HTTPREQUEST_ERROR;
  }

  if (TTV_SUCCEEDED(ec)) {
    mLibNetMemId = sceNetPoolCreate("http", kNetHeapSize, 0);
    if (mLibNetMemId < 0) {
      // SCE_NET_ERROR_EINVAL 0x80410116 Function called with an invalid argument or content
      // SCE_NET_ERROR_ENFILE 0x80410117 Limit for creatable memory pools was reached
      // SCE_NET_ERROR_ENOALLOCMEM 0x804101ce Memory could not be allocated
      // SCE_NET_ERROR_ENAMETOOLONG 0x8041013f The debug name is too long
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceNetPoolCreate() failed (0x%x)", mLibNetMemId);
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    mLibSslCtxId = sceSslInit(kSslHeapSize);
    if (mLibSslCtxId < 0) {
      // SCE_SSL_ERROR_OUT_OF_MEMORY 0x809517D5 Memory pool allocation failed
      // SCE_SSL_ERROR_OUT_OF_SIZE 0x8095F008 There are insufficient library context IDs
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceSslInit() failed (0x%x)", mLibSslCtxId);
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    mLibHttpCtxId = sceHttpInit(mLibNetMemId, mLibSslCtxId, kHttpHeapSize);
    if (mLibHttpCtxId < 0) {
      // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpInit() failed (0x%x)", mLibHttpCtxId);
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    mInitialized = true;
  } else {
    if (mLibNetMemId >= 0) {
      sceNetPoolDestroy(mLibNetMemId);
    }

    if (mLibSslCtxId >= 1) {
      sceSslTerm(mLibSslCtxId);
    }

    assert(mLibHttpCtxId < 0);

    mLibNetMemId = -1;
    mLibSslCtxId = -1;
    mLibHttpCtxId = -1;
  }

  return ec;
}

TTV_ErrorCode ttv::OrbisHttpRequest::Initialize(int libNetMemId, int libSslCtxId, int libHttpCtxId) {
  if (mInitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  mLibNetMemId = libNetMemId;
  mLibSslCtxId = libSslCtxId;
  mLibHttpCtxId = libHttpCtxId;
  mInitialized = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::OrbisHttpRequest::Shutdown() {
  if (!mInitialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (mLibHttpCtxId >= 1) {
    sceHttpTerm(mLibHttpCtxId);
    mLibHttpCtxId = -1;
  }

  if (mLibSslCtxId >= 1) {
    sceSslTerm(mLibSslCtxId);
    mLibSslCtxId = -1;
  }

  if (mLibNetMemId >= 0) {
    sceNetPoolDestroy(mLibNetMemId);
    mLibNetMemId = -1;
  }

  mInitialized = false;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::OrbisHttpRequest::SendHttpRequest(const std::string& /*requestName*/, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData) {
  if (!mInitialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  if (url.empty()) {
    return TTV_EC_INVALID_HTTP_REQUEST_PARAMS;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // Create the template
  int templateId = 0;
  int sceRet = sceHttpCreateTemplate(mLibHttpCtxId, kPs4UserAgent, SCE_HTTP_VERSION_1_1, SCE_TRUE);
  if (sceRet < 0) {
    // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
    // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
    // SCE_HTTP_ERROR_INVALID_VERSION 0x8043106a The HTTP version is invalid
    // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
    // SCE_HTTP_ERROR_INVALID_ID 0X80431100 Invalid Http library context ID
    // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
    ec = TTV_EC_HTTPREQUEST_ERROR;
    ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpCreateTemplate() failed (0x%x)", sceRet);
  } else {
    templateId = sceRet;
  }

  // Create the connection
  int connectionId = 0;
  if (TTV_SUCCEEDED(ec)) {
    sceRet = sceHttpCreateConnectionWithURL(templateId, url.c_str(), SCE_TRUE);
    if (sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
      // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
      // SCE_HTTP_ERROR_UNKNOWN_SCHEME 0x80431061 A scheme other than HTTP or HTTPS was specified in the URI
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The specified template ID is invalid
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error,
        "sceHttpCreateConnectionWithURL() failed (0x%x) with Url: %s", sceRet, url.c_str());
    } else {
      connectionId = sceRet;
    }
  }

  // Create request
  int requestId = 0;
  if (TTV_SUCCEEDED(ec)) {
    switch (httpReqType) {
      case HTTP_GET_REQUEST:
        sceRet = sceHttpCreateRequestWithURL(connectionId, SCE_HTTP_METHOD_GET, url.c_str(), 0);
        break;
      case HTTP_PUT_REQUEST:
        sceRet = sceHttpCreateRequestWithURL(connectionId, SCE_HTTP_METHOD_PUT, url.c_str(), requestBodySize);
        break;
      case HTTP_POST_REQUEST:
        sceRet = sceHttpCreateRequestWithURL(connectionId, SCE_HTTP_METHOD_POST, url.c_str(), requestBodySize);
        break;
      case HTTP_DELETE_REQUEST:
        sceRet = sceHttpCreateRequestWithURL(connectionId, SCE_HTTP_METHOD_DELETE, url.c_str(), requestBodySize);
        break;
      default:
        ec = TTV_EC_HTTPREQUEST_ERROR;
        break;
    }

    if (TTV_FAILED(ec) || sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
      // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
      // SCE_HTTP_ERROR_INVALID_VERSION 0x8043106a PUT or DELETE was set for method using connection settings when 1.0
      // was set as the HTTP version SCE_HTTP_ERROR_UNKNOWN_METHOD 0x8043106b The value specified in method is invalid
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The specified ID of the connection settings is invalid
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error,
        "sceHttpCreateRequestWithURL() failed (0x%x) with Url: %s", sceRet, url.c_str());
    } else {
      requestId = sceRet;
    }
  }

  // Add request headers
  if (TTV_SUCCEEDED(ec)) {
    for (const auto& header : requestHeaders) {
      sceRet =
        sceHttpAddRequestHeader(requestId, header.paramName.c_str(), header.paramValue.c_str(), SCE_HTTP_HEADER_ADD);
      if (sceRet < 0) {
        // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
        // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
        // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
        // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
        // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe name is NULL or the value specified for mode is invalid
        ec = TTV_EC_HTTPREQUEST_ERROR;
        ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error,
          "sceHttpAddRequestHeader() failed (0x%x) with Header - %s:%s", sceRet, header.paramName.c_str(),
          header.paramValue.c_str());

        break;
      }
    }
  }

  // Send the request
  if (TTV_SUCCEEDED(ec)) {
    if (httpReqType == HTTP_GET_REQUEST) {
      sceRet = sceHttpSendRequest(requestId, NULL, 0);
    } else {
      sceRet = sceHttpSendRequest(requestId, requestBody, requestBodySize);
    }

    if (sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 One of these three has occurred:
      // Multiple threads attempted to send requests simultaneously using the same connection settings
      // Attempted to send the next request using the same connection settings before sceHttpReadData() finished
      // receiving data Attempted to send another request using the same connection settings before the sending of POST
      // data completed SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
      // SCE_HTTP_ERROR_UNKNOWN_SCHEME 0x80431061 A scheme other than HTTP or HTTPS was detected during request
      // processing SCE_HTTP_ERROR_NETWORK 0x80431063 An error was returned by the TCP stack SCE_HTTP_ERROR_BAD_RESPONSE
      // 0x80431064 Response from server is invalid SCE_HTTP_ERROR_AFTER_SEND 0x80431066 A request that has already had
      // its request sending complete was specified SCE_HTTP_ERROR_TIMEOUT 0x80431068 Either the timeout period set
      // using the timeout setting function has passed or the TCP timeout period has passed.
      // SCE_HTTP_ERROR_UNKNOWN_METHOD 0x8043106b The method specified for the request is invalid
      // SCE_HTTP_ERROR_SSL 0x80431075 SSL communication was failed
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_ABORTED 0x80431080 Request was aborted
      // SCE_HTTP_ERROR_EAGAIN 0x80431082 Request is blocked
      // SCE_HTTP_ERROR_PROXY 0x80431084 Failed to establish the connection to the HTTP Proxy
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
      // SCE_HTTP_ERROR_OUT_OF_SIZE 0x80431104 Response exceeding the sizes that the library can handle was received
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      // SCE_HTTP_ERROR_INVALID_URL 0x80433060 Invalid URL was detected during request processing
      // SCE_SSL_ERROR_INVALID_CERT 0x8095F00A Format of the server certificate is invalid
      // SCE_SSL_ERROR_CN_CHECK 0x8095F00B Common name or subject alternative names of the server certificate does not
      // match the host name SCE_SSL_ERROR_UNKNOWN_CA 0x8095F00C Certificate of RootCA that issued the server
      // certificate is not included in the locally held certificates SCE_SSL_ERROR_NOT_AFTER_CHECK 0x8095F00D Validity
      // period of the server certificate expired SCE_SSL_ERROR_NOT_BEFORE_CHECK 0x8095F00E Validity period of the
      // server certificate has not started yet
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpSendRequest() failed (0x%x)", sceRet);
    }
  }

  // Read the status code
  int statusCode = 0;
  if (TTV_SUCCEEDED(ec)) {
    sceRet = sceHttpGetStatusCode(requestId, &statusCode);
    if (sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
      // SCE_HTTP_ERROR_BEFORE_SEND 0x80431065 The specified request has not been sent yet
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpGetStatusCode() failed (0x%x)", sceRet);
    }
  }

  // Get response headers
  char* headerData = nullptr;
  size_t headerSize = 0;
  bool notifyResponse = true;
  std::map<std::string, std::string> responseHeaders;
  if (TTV_SUCCEEDED(ec)) {
    sceRet = sceHttpGetAllResponseHeaders(requestId, &headerData, &headerSize);
    if (sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
      // SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
      // SCE_HTTP_ERROR_BEFORE_SEND 0x80431065 The specified request has not been sent yet
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message(
        "OrbisHttpRequest", MessageLevel::Error, "sceHttpGetAllResponseHeaders() failed (0x%x)", sceRet);
    } else {
      if (headerSize > 0) {
        std::vector<std::string> headerLines;
        ttv::Split(headerData, "\r\n", headerLines);

        for (const auto& headerLine : headerLines) {
          size_t index = headerLine.find(':');
          if (index != std::string::npos) {
            std::string name = headerLine.substr(0, index);
            std::string value = headerLine.substr(index + 1);
            ttv::Trim(value);
            responseHeaders[name] = value;
          }
        }
      }

      // Send headers to the client and see if it wants the body
      if (headersCallback != nullptr) {
        notifyResponse = headersCallback(static_cast<uint>(statusCode), responseHeaders, userData);
      }
    }
  }

  // Send body to client
  if (TTV_SUCCEEDED(ec) && notifyResponse) {
    uint64_t contentLength = 0;
    int contentLengthType = 0;

    sceRet = sceHttpGetResponseContentLength(requestId, &contentLengthType, &contentLength);
    if (sceRet < 0) {
      // SCE_HTTP_ERROR_BUSY 0x80431021 An ID being deleted by another thread was specified
      // SCE_HTTP_ERROR_BEFORE_SEND 0x80431065 The specified request has not been sent yet
      // SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
      // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
      // SCE_HTTP_ERROR_INVALID_VALUE 0x804311fe An invalid value was specified for an argument
      ec = TTV_EC_HTTPREQUEST_ERROR;
      ttv::trace::Message(
        "OrbisHttpRequest", MessageLevel::Error, "sceHttpGetResponseContentLength() failed (0x%x)", sceRet);
    } else {
      std::vector<char> response;

      if (contentLengthType == SCE_HTTP_CONTENTLEN_EXIST) {
        // Copy response data directly into vector
        response.resize(contentLength);
        sceRet = sceHttpReadData(requestId, response.data(), contentLength);
        if (sceRet < 0) {
          // SCE_HTTP_ERROR_BUSY 0x80431021 Multiple threads attempted to send requests simultaneously using the same
          // connection settings SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
          // SCE_HTTP_ERROR_BAD_RESPONSE 0x80431064 Response from server is invalid
          // SCE_HTTP_ERROR_BEFORE_SEND 0x80431065 The specified request has not been sent yet
          // SCE_HTTP_ERROR_TIMEOUT 0x80431068 Either the timeout period set using the timeout setting function has
          // passed or the TCP timeout period has passed. SCE_HTTP_ERROR_READ_BY_HEAD_METHOD 0x8043106f The specified
          // request is for the HEAD method SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
          // SCE_HTTP_ERROR_ABORTED 0x80431080 Request was aborted
          // SCE_HTTP_ERROR_EAGAIN 0x80431082 Request is blocked
          // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
          // SCE_HTTP_ERROR_OUT_OF_SIZE 0x80431104 Response exceeding the sizes that the library can handle was received
          ec = TTV_EC_HTTPREQUEST_ERROR;
          ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpReadData() failed (0x%x)", sceRet);
        }
      } else {
        // Build up the entire body through a temporary buffer
        const uint kBufferSize = 1024;
        char buffer[kBufferSize];
        bool receivingData = true;

        while (receivingData) {
          sceRet = sceHttpReadData(requestId, buffer, kBufferSize);
          response.insert(response.cend(), buffer, buffer + sceRet);
          if (sceRet < 0) {
            // SCE_HTTP_ERROR_BUSY 0x80431021 Multiple threads attempted to send requests simultaneously using the same
            // connection settings SCE_HTTP_ERROR_OUT_OF_MEMORY 0x80431022 Insufficient free memory space
            // SCE_HTTP_ERROR_BAD_RESPONSE 0x80431064 Response from server is invalid
            // SCE_HTTP_ERROR_BEFORE_SEND 0x80431065 The specified request has not been sent yet
            // SCE_HTTP_ERROR_TIMEOUT 0x80431068 Either the timeout period set using the timeout setting function has
            // passed or the TCP timeout period has passed. SCE_HTTP_ERROR_READ_BY_HEAD_METHOD 0x8043106f The specified
            // request is for the HEAD method SCE_HTTP_ERROR_INSUFFICIENT_STACKSIZE 0x80431076 Insufficient stack size
            // SCE_HTTP_ERROR_ABORTED 0x80431080 Request was aborted
            // SCE_HTTP_ERROR_EAGAIN 0x80431082 Request is blocked
            // SCE_HTTP_ERROR_INVALID_ID 0x80431100 The ID specified for the argument is invalid
            // SCE_HTTP_ERROR_OUT_OF_SIZE 0x80431104 Response exceeding the sizes that the library can handle was
            // received
            ec = TTV_EC_HTTPREQUEST_ERROR;
            ttv::trace::Message("OrbisHttpRequest", MessageLevel::Error, "sceHttpReadData() failed (0x%x)", sceRet);
            break;
          } else if (sceRet == 0) {
            // Done receiving data
            receivingData = false;
          }
        }
      }

      // Notify the client
      responseCallback(static_cast<uint>(statusCode), response, userData);
    }
  }

  if (requestId != 0) {
    sceHttpDeleteRequest(requestId);
  }

  if (connectionId != 0) {
    sceHttpDeleteConnection(connectionId);
  }

  if (templateId != 0) {
    sceHttpDeleteTemplate(templateId);
  }

  return ec;
}
