/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/random.h"

#include <libsecure.h>
#include <sce_random.h>

namespace ttv {
namespace random {
namespace {
/**
 * Orbis Random Generator
 * This use the SceLibSecure, which provides cryptographically secure random generation
 * sceSysmoduleLoadModule(SCE_SYSMODULE_RANDOM) must be done by the app before this can be used
 * The app also needs to link with libSceRandom_stub_weak.a
 */

/*
 * This provides a one time initialization of the orbis random functionality
 * across all thread and a thread safe access to its helper functions
 */
class OrbisRandomAdaptor {
 private:
  OrbisRandomAdaptor() {
    TTV_ErrorCode ret = ttv::CreateMutex(mMutex, "ThreadSafeOrbisGenerator");
    ASSERT_ON_ERROR(ret);

    Initialize();
  }

 public:
  static OrbisRandomAdaptor& Get() {
    static OrbisRandomAdaptor instance;
    return instance;
  }

  size_t FillRandomData(char* data, size_t length) {
    // sceLibSecureInit can only be initialized once for all threads which is used by sceLibSecureRandom.  Cannot create
    // new instances between threads since that will share non thread safe methods.  Mutex needed to ensure thread safe
    // random number generation
    AutoMutex lock(mMutex.get());

    SceLibSecureBlock memBlock = {data, length};
    SceLibSecureErrorType ret = sceLibSecureRandom(&memBlock);

    if (ret != SCE_LIBSECURE_OK) {
      TTV_ASSERT(false);
      return 0;
    } else {
      return length;
    }
  }

 private:
  void Initialize() {
    for (size_t index = 0; index < sizeof(mSeed);) {
      size_t remaining = sizeof(mSeed) - index;
      size_t toFill = std::min(remaining, static_cast<size_t>(SCE_RANDOM_MAX_SIZE));

      // Seed the random generator with high-entropy data
      int ret = sceRandomGetRandomNumber(&mSeed[index], toFill);
      TTV_ASSERT(ret == SCE_OK);

      index += toFill;
    }

    SceLibSecureBlock memBlock = {mSeed, sizeof(mSeed)};
    SceLibSecureErrorType ret = sceLibSecureInit(SCE_LIBSECURE_FLAGS_RANDOM_GENERATOR, &memBlock);
    TTV_ASSERT(ret == SCE_LIBSECURE_OK);
  }

  char mSeed[10 * 1024];  // High-entropy random data used for initial seed

  std::unique_ptr<ttv::IMutex> mMutex;
};

class OrbisGenerator : public IGenerator {
 public:
  OrbisGenerator() = default;

  virtual result_type operator()() override {
    if (mRemaining <= 0) {
      TTV_ASSERT(mRemaining == 0);
      mRemaining = OrbisRandomAdaptor::Get().FillRandomData(mData, sizeof(mData));
    }

    TTV_ASSERT(mRemaining >= sizeof(result_type));

    size_t index = sizeof(mData) - mRemaining;
    mRemaining -= sizeof(result_type);

    return *reinterpret_cast<result_type*>(&mData[index]);
  }

 private:
  char mData[1024];       // Current generated random data
  size_t mRemaining = 0;  // How much data is left to be read in randomData
};

class ThreadSafeOrbisGenerator : public ttv::random::IGenerator {
 public:
  ThreadSafeOrbisGenerator() = default;

  virtual result_type operator()() final {
    // We want to ensure that only one generator exists per thread.  This
    // will create a new Generator object anytime a unique thread calls this
    // method.
    static thread_local OrbisGenerator engine;
    return engine();
  }
};

}  // namespace

IGenerator& GetGenerator() {
  static ThreadSafeOrbisGenerator engine;
  return engine;
}
}  // namespace random
}  // namespace ttv
