##########################################################################################################
#
# Fragment: core_std
#
# See README.md for fragment documentation.
#
##########################################################################################################

import glob
import os
import build_types

def load_twitchsdk_fragment(primitives, options):

    fragment = build_types.SourceFragment('twitchsdk', 'core_std', os.path.dirname(os.path.realpath(__file__)))

    # Pull in primitives
    use_all_implementations = options.has_setting('TWITCH_USE_ALL_STD_IMPLEMENTATIONS')

    if not primitives.implemented('MEMORY'):
        fragment.add_source_files( os.path.join(fragment.root_path, 'source/standardmemory.cpp') )
        primitives.set('MEMORY', None)

    if not primitives.implemented('ASSERT'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardassert.cpp") )
        primitives.set('ASSERT', None)

    if not primitives.implemented('RANDOM'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardrandom.cpp") )
        primitives.set('RANDOM', None)

    if use_all_implementations or options.has_setting('TTV_USE_STD_MUTEX'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardmutex.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardmutex.h") )

        if not primitives.implemented('MUTEX_FACTORY'):
            primitives.set('MUTEX_FACTORY', "ttv::StandardMutexFactory", "twitchsdk/core/standardmutex.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_RAW_SOCKET'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardsocket.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardsocket.h") )

        if not primitives.implemented('RAW_SOCKET_FACTORY'):
            primitives.set('RAW_SOCKET_FACTORY', "ttv::StandardSocketFactory", "twitchsdk/core/standardsocket.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_OPENSSL_SOCKET'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardopensslsocket.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardopensslsocket.h") )

        # Flag
        flag_code_lines = [
            "bool gStandardOpenSslInitialized = false;"
        ]

        # Init code
        init_code_lines = [
            "TTV_ErrorCode ec = ttv::OpenSslSocket::InitializeOpenSslSockets();",
            "gStandardOpenSslInitialized = TTV_SUCCEEDED(ec);"
        ]

        # Shutdown code
        shutdown_code_lines = [
            "if (gStandardOpenSslInitialized)",
            "{",
            "    gStandardOpenSslInitialized = false;",
            "    ttv::OpenSslSocket::ShutdownOpenSslSockets();",
            "}"
        ]

        if not primitives.implemented('TLS_SOCKET_FACTORY'):
            primitives.set('TLS_SOCKET_FACTORY', "ttv::OpenSslSocketFactory", "twitchsdk/core/standardopensslsocket.h", flag_code_lines, init_code_lines, shutdown_code_lines)

        # Dependencies
        fragment.add_header_search_paths( os.path.join(options.common_dependencies_path, 'openssl', 'include', options.compiler, options.architecture) )
        options.platform_settings.link_against_openssl(options, fragment)

    if use_all_implementations or options.has_setting('TTV_USE_STD_WEBSOCKET'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardwebsocket.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardwebsocket.h") )

        if not primitives.implemented('WS_SOCKET_FACTORY'):
            primitives.set('WS_SOCKET_FACTORY', "ttv::StandardWebSocketFactory", "twitchsdk/core/standardwebsocket.h")

        # Preprocessor defines
        if options.compiler == "vs2013":
            fragment.add_preprocessor_definition('_WEBSOCKETPP_CPP11_THREAD_')
        else:
            fragment.add_preprocessor_definition('_WEBSOCKETPP_CPP11_STL_')

        # Dependencies
        fragment.add_dependency_search_paths( os.path.join(options.common_dependencies_path, 'websocketpp') )

    if use_all_implementations or options.has_setting('TTV_USE_STD_FILE_SOCKET'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardfilesocket.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardfilesocket.h") )

    if use_all_implementations or options.has_setting('TTV_USE_STD_SYSTEM_CLOCK'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardsystemclock.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardsystemclock.h") )

        if not primitives.implemented('SYSTEM_CLOCK'):
            primitives.set('SYSTEM_CLOCK', "ttv::StandardSystemClock", "twitchsdk/core/standardsystemclock.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_THREAD_SYNC'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardthreadsync.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardthreadsync.h") )

        if not primitives.implemented('THREAD_SYNC_FACTORY'):
            primitives.set('THREAD_SYNC_FACTORY', "ttv::StandardThreadSyncFactory", "twitchsdk/core/standardthreadsync.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_BACKGROUND_EVENT_SCHEDULER'):
        if not primitives.implemented('BACKGROUND_EVENTSCHEDULER_FACTORY'):
            primitives.set('BACKGROUND_EVENTSCHEDULER_FACTORY', "ttv::ThreadedEventSchedulerFactory", "twitchsdk/core/threadedeventschedulerfactory.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_THREAD'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardthread.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardthread.h") )

        if not primitives.implemented('THREAD_FACTORY'):
            primitives.set('THREAD_FACTORY', "ttv::StandardThreadFactory", "twitchsdk/core/standardthread.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_THREAD_API'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardthreadapi.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardthreadapi.h") )

        if not primitives.implemented('THREAD_API'):
            primitives.set('THREAD_API', "ttv::StandardThreadApi", "twitchsdk/core/standardthreadapi.h")

    if use_all_implementations or options.has_setting('TTV_USE_STD_TRACER'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/standardtracer.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/standardtracer.h") )

        if not primitives.implemented('TRACER'):
            primitives.set('TRACER', "ttv::StandardTracer", "twitchsdk/core/standardtracer.h")

    fragment.add_source_group("Source Files/core/std", fragment.source_files)
    fragment.add_source_group("Header Files/core/std", fragment.header_files)

    # Include paths
    fragment.add_header_search_paths( os.path.join(fragment.root_path, 'include') )

    return fragment
