/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2015 Twitch Interactive, Inc.
 *********************************************************************************************/

#pragma once

#include "twitchsdk/core/mutex.h"

#include <condition_variable>

#include <mutex>
#include <string>

namespace ttv {
class StandardMutex;
class StandardRecursiveMutex;
class StandardConditionMutex;
class StandardMutexFactory;

template <typename MutexType>
class StandardMutexBase;
}  // namespace ttv

template <typename MutexType>
class ttv::StandardMutexBase : public IMutex {
 public:
  StandardMutexBase(const std::string& name = "") : mName(name) {}

  virtual ~StandardMutexBase() override = default;

  virtual TTV_ErrorCode Lock() override {
    mMutex.lock();

    return TTV_EC_SUCCESS;
  }

  virtual TTV_ErrorCode TryLock() override {
    TTV_ErrorCode ret = TTV_EC_SUCCESS;

    const bool acquiredLock = mMutex.try_lock();
    if (!acquiredLock) {
      ret = TTV_EC_MUTEX_NOT_ACQUIRED;
    }

    return ret;
  }

  virtual TTV_ErrorCode Unlock() override {
    mMutex.unlock();

    return TTV_EC_SUCCESS;
  }

 private:
  MutexType mMutex;
  std::string mName;
};

class ttv::StandardMutex : public StandardMutexBase<std::mutex> {
 public:
  StandardMutex(const std::string& name = "") : StandardMutexBase<std::mutex>(name) {}
};

class ttv::StandardRecursiveMutex : public StandardMutexBase<std::recursive_mutex> {
 public:
  StandardRecursiveMutex(const std::string& name = "") : StandardMutexBase<std::recursive_mutex>(name) {}
};

class ttv::StandardConditionMutex : public IConditionMutex {
 public:
  StandardConditionMutex(const std::string& name = "");
  virtual ~StandardConditionMutex() override = default;

  virtual TTV_ErrorCode Lock() override;
  virtual TTV_ErrorCode TryLock() override;
  virtual TTV_ErrorCode Unlock() override;
  virtual TTV_ErrorCode Wait() override;
  virtual TTV_ErrorCode WaitFor(uint64_t timeoutMilliseconds) override;
  virtual TTV_ErrorCode Signal() override;
  virtual TTV_ErrorCode Broadcast() override;

 private:
  std::mutex mMutex;
  std::condition_variable mCondition;
  std::string mName;
};

class ttv::StandardMutexFactory : public IMutexFactory {
 public:
  virtual TTV_ErrorCode CreateMutex(std::unique_ptr<IMutex>& result, const std::string& name = "") override;
  virtual TTV_ErrorCode CreateConditionMutex(
    std::unique_ptr<IConditionMutex>& result, const std::string& name = "") override;
};
