/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/socket.h"

#include <memory>

namespace ttv {
class OpenSslSocket;
class OpenSslSocketFactory;
}  // namespace ttv

/**
 * This secure socket implements platform-independent TLS communication.  Note that it only supports non-blocking IO.
 */
class ttv::OpenSslSocket : public ISocket {
 public:
  OpenSslSocket();
  ~OpenSslSocket();

  TTV_ErrorCode Initialize(const std::string& host, const std::string& port);

  // ISocket Implementation
  virtual TTV_ErrorCode Connect();
  virtual TTV_ErrorCode Disconnect();
  virtual TTV_ErrorCode Send(const uint8_t* buffer, size_t length, size_t& sent);
  virtual TTV_ErrorCode Recv(uint8_t* buffer, size_t length, size_t& received);
  virtual uint64_t TotalSent();
  virtual uint64_t TotalReceived();
  virtual bool Connected();

  TTV_ErrorCode SetCertificateData(const std::string& data);

  static TTV_ErrorCode InitializeOpenSslSockets();
  static TTV_ErrorCode ShutdownOpenSslSockets();

  /**
   * Pass in trusted hosts that we allow SSL connections to.
   * @param[in] List of strings of hosts that we can trust and verify against.
   */
  static void SetTrustedHosts(const std::vector<std::string>& trustedHosts);

 private:
  class SocketData;

  TTV_ErrorCode PrepareConnection();
  TTV_ErrorCode Handshake();
  TTV_ErrorCode Flush();
  TTV_ErrorCode FlushOutgoing();
  TTV_ErrorCode FlushIncoming();

  static std::vector<std::string> sTrustedHosts;
  std::string mHost;
  std::string mPort;
  std::shared_ptr<ISocket> mBaseSocket;
  std::unique_ptr<SocketData> mSocketData;
  std::string mCertificateData;

  bool mConnected;
};

class ttv::OpenSslSocketFactory : public ttv::ISocketFactory {
 public:
  virtual ~OpenSslSocketFactory();
  virtual bool IsProtocolSupported(const std::string& protocol);
  virtual TTV_ErrorCode CreateSocket(const std::string& uri, std::shared_ptr<ISocket>& result);
};
