/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/standardfilesocket.h"

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/httprequestutils.h"

ttv::StandardFileSocket::StandardFileSocket()
    : mInputFile(nullptr), mOutputFile(nullptr), mTotalSent(0), mTotalRecieved(0), mConnected(false) {}

ttv::StandardFileSocket::~StandardFileSocket() {
  (void)Disconnect();
}

TTV_ErrorCode ttv::StandardFileSocket::Initialize(const std::string& inputPath, const std::string& outputPath) {
  mInputFilePath = inputPath;
  mOutputFilePath = outputPath;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardFileSocket::Connect() {
  TTV_ASSERT(!mConnected);
  if (mConnected) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mTotalSent = 0;
  mTotalRecieved = 0;
  mConnected = true;

  // Open the files
  if (mInputFilePath != "") {
    mInputFile = fopen(mInputFilePath.c_str(), "rb");
    if (mInputFile == nullptr) {
      TTV_ASSERT(false);
      return TTV_EC_INVALID_ARG;
    }
  }

  if (mOutputFilePath != "") {
    mOutputFile = fopen(mOutputFilePath.c_str(), "wb");
    if (mOutputFile == nullptr) {
      TTV_ASSERT(false);
      Disconnect();
      return TTV_EC_INVALID_ARG;
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardFileSocket::Disconnect() {
  // Close files
  mConnected = false;

  if (mInputFile != nullptr) {
    (void)fclose(mInputFile);
    mInputFile = nullptr;
  }

  if (mOutputFile != nullptr) {
    (void)fclose(mOutputFile);
    mOutputFile = nullptr;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardFileSocket::Send(const uint8_t* buffer, size_t length, size_t& sent) {
  TTV_ASSERT(buffer);
  TTV_ASSERT(length > 0);

  sent = 0;

  if (!Connected()) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  if (mOutputFile != nullptr) {
    sent = fwrite(buffer, length, 1, mOutputFile);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardFileSocket::Recv(uint8_t* buffer, size_t length, size_t& received) {
  TTV_ASSERT(buffer);

  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  received = 0;

  if (Connected()) {
    if (mInputFile != nullptr) {
      received = fread(buffer, 1, length, mInputFile);
    }
  } else {
    ret = TTV_EC_SOCKET_ENOTCONN;
  }

  return TTV_EC_SUCCESS;
}

uint64_t ttv::StandardFileSocket::TotalSent() {
  return mTotalSent;
}

uint64_t ttv::StandardFileSocket::TotalReceived() {
  return mTotalRecieved;
}

bool ttv::StandardFileSocket::Connected() {
  return mConnected;
}

ttv::StandardFileSocketFactory::~StandardFileSocketFactory() {}

bool ttv::StandardFileSocketFactory::IsProtocolSupported(const std::string& protocol) {
  return protocol == "" || protocol == "tcp";
}

TTV_ErrorCode ttv::StandardFileSocketFactory::CreateSocket(
  const std::string& uri, std::shared_ptr<ttv::ISocket>& result) {
  result.reset();

  Uri url(uri);
  if (url.GetProtocol() == "file") {
    // TODO: Parse the uri to get the paths
    std::shared_ptr<StandardFileSocket> socket = std::make_shared<StandardFileSocket>();
    if (TTV_SUCCEEDED(socket->Initialize("", ""))) {
      result = socket;
      return TTV_EC_SUCCESS;
    }
  }

  return TTV_EC_UNIMPLEMENTED;
}
