/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/standardmutex.h"

ttv::StandardConditionMutex::StandardConditionMutex(const std::string& name) : mName(name) {}

TTV_ErrorCode ttv::StandardConditionMutex::Lock() {
  mMutex.lock();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardConditionMutex::TryLock() {
  const bool acquiredLock = mMutex.try_lock();
  if (!acquiredLock) {
    return TTV_EC_MUTEX_NOT_ACQUIRED;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardConditionMutex::Unlock() {
  mMutex.unlock();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardConditionMutex::Wait() {
  std::unique_lock<std::mutex> lock(mMutex, std::adopt_lock);
  mCondition.wait(lock);
  lock.release();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardConditionMutex::WaitFor(uint64_t timeoutMilliseconds) {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  std::unique_lock<std::mutex> lock(mMutex, std::adopt_lock);
  const bool timedOut =
    mCondition.wait_for(lock, std::chrono::milliseconds(timeoutMilliseconds)) == std::cv_status::timeout;
  lock.release();

  if (timedOut) {
    ret = TTV_EC_WAIT_TIMEOUT;
  }

  return ret;
}

TTV_ErrorCode ttv::StandardConditionMutex::Signal() {
  mCondition.notify_one();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardConditionMutex::Broadcast() {
  mCondition.notify_all();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardMutexFactory::CreateMutex(std::unique_ptr<ttv::IMutex>& result, const std::string& name) {
  result = std::make_unique<StandardMutex>(name);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::StandardMutexFactory::CreateConditionMutex(
  std::unique_ptr<ttv::IConditionMutex>& result, const std::string& name) {
  result = std::make_unique<StandardConditionMutex>(name);

  return TTV_EC_SUCCESS;
}
