##########################################################################################################
#
# Fragment: core_win32
#
# See README.md for fragment documentation.
#
##########################################################################################################

import glob
import os
import build_types

def load_twitchsdk_fragment(primitives, options):

    fragment = build_types.SourceFragment('twitchsdk', 'core_win32', os.path.dirname(os.path.realpath(__file__)))

    # Preprocessor defines
    fragment.add_preprocessor_definition('TTV_CORE_INCLUDE_PCH_PLATFORM')

    # Use custom implementation of certain C++11 features
    if options.compiler == "vs2010" or options.compiler == "vs2012":
        fragment.add_preprocessor_definition('TTV_INCLUDE_CPPTRANSITIONS')

    # Pull in primitives
    use_all_implementations = options.has_setting('TTV_USEALL_WIN32_IMPLEMENTATIONS')

    if not primitives.implemented('MEMORY'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winmemory.cpp") )
        primitives.set('MEMORY', None)

    if not primitives.implemented('ASSERT'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winassert.cpp") )
        primitives.set('ASSERT', None)

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_MUTEX'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winmutex.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/winmutex.h") )

        if not primitives.implemented('MUTEX_FACTORY'):
            primitives.set('MUTEX_FACTORY', "ttv::WinMutexFactory", "twitchsdk/core/winmutex.h")

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_RAW_SOCKET'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winsocket.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/winsocket.h") )

        # Flag
        flag_code_lines = [
            "bool gWinsockInitialized = false;"
        ]

        # Init code
        init_code_lines = [
            "TTV_ErrorCode ec = ttv::WinSocket::InitializeWinSock();",
            "gWinsockInitialized = TTV_SUCCEEDED(ec);"
        ]

        # Shutdown code
        shutdown_code_lines = [
            "if (gWinsockInitialized)",
            "{",
            "    gWinsockInitialized = false;",
            "    ttv::WinSocket::ShutdownWinSock();",
            "}"
        ]

        if not primitives.implemented('RAW_SOCKET_FACTORY'):
            primitives.set('RAW_SOCKET_FACTORY', "ttv::WinSocketFactory", "twitchsdk/core/winsocket.h", flag_code_lines, init_code_lines, shutdown_code_lines)

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_SYSTEM_CLOCK'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winsystemclock.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/winsystemclock.h") )

        if not primitives.implemented('SYSTEM_CLOCK'):
            primitives.set('SYSTEM_CLOCK', "ttv::WinSystemClock", "twitchsdk/core/winsystemclock.h")

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_THREAD_SYNC'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winthreadsync.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/winthreadsync.h") )

        if not primitives.implemented('THREAD_SYNC_FACTORY'):
            primitives.set('THREAD_SYNC_FACTORY', "ttv::WinThreadSyncFactory", "twitchsdk/core/winthreadsync.h")

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_THREAD'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/winthread.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/winthread.h") )

        if not primitives.implemented('THREAD_FACTORY'):
            primitives.set('THREAD_FACTORY', "ttv::WinThreadFactory", "twitchsdk/core/winthread.h")

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_HTTPREQUEST'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/wininethttprequest.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/wininethttprequest.h") )

        if not primitives.implemented('HTTP_REQUEST'):
            primitives.set('HTTP_REQUEST', "ttv::WinInetHttpRequest", "twitchsdk/core/wininethttprequest.h")

    if use_all_implementations or options.has_setting('TTV_USE_WIN32_TRACER'):
        fragment.add_source_files( os.path.join(fragment.root_path, "source/wintracer.cpp") )
        fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/wintracer.h") )

        if not primitives.implemented('TRACER'):
            primitives.set('TRACER', "ttv::WinTracer", "twitchsdk/core/wintracer.h")

    fragment.add_header_files( os.path.join(fragment.root_path, "include/twitchsdk/core/internal/pch_platform.h") )

    fragment.add_source_group("Source Files/core/win32", fragment.source_files)
    fragment.add_source_group("Header Files/core/win32", fragment.header_files)

    # Include paths
    fragment.add_header_search_paths( os.path.join(fragment.root_path, 'include') )

    return fragment
