/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/mutex.h"

#include <windows.h>

#include <string>

namespace ttv {
class WinMutex;
class WinConditionMutex;
class WinMutexFactory;
}  // namespace ttv

/**
 * Windows implementation of IMutex using WinApi Critical Section.
 * https://msdn.microsoft.com/en-us/library/windows/desktop/ms682530(v=vs.85).aspx.
 * Windows Server 2003 / XP and below aren't supported.
 */
class ttv::WinMutex : public IMutex {
 public:
  WinMutex(const std::string& name = "");
  virtual ~WinMutex() override;

  virtual TTV_ErrorCode Lock() override;
  virtual TTV_ErrorCode TryLock() override;
  virtual TTV_ErrorCode Unlock() override;

 private:
  TTV_ErrorCode Destroy();

  CRITICAL_SECTION mCriticalSection;
  bool mLocked;
};

/**
 * Windows implementation of IConditionMutex using WinApi Condition Variables.
 * https://msdn.microsoft.com/en-us/library/windows/desktop/ms682052(v=vs.85).aspx.
 * Windows Server 2003 / XP and below aren't supported.
 */
class ttv::WinConditionMutex : public IConditionMutex {
 public:
  WinConditionMutex(const std::string& name = "");
  virtual ~WinConditionMutex() override;

  virtual TTV_ErrorCode Lock() override;
  virtual TTV_ErrorCode TryLock() override;
  virtual TTV_ErrorCode Unlock() override;
  virtual TTV_ErrorCode Wait() override;
  virtual TTV_ErrorCode WaitFor(uint64_t timeoutMilliseconds) override;
  virtual TTV_ErrorCode Signal() override;
  virtual TTV_ErrorCode Broadcast() override;

 private:
  TTV_ErrorCode Destroy();

  CRITICAL_SECTION mCriticalSection;
  CONDITION_VARIABLE mConditionVariable;
  bool mLocked;
};

class ttv::WinMutexFactory : public IMutexFactory {
 public:
#pragma push_macro("CreateMutex")
#undef CreateMutex
  virtual TTV_ErrorCode CreateMutex(std::unique_ptr<IMutex>& result, const std::string& name = "") override;
#pragma pop_macro("CreateMutex")

  virtual TTV_ErrorCode CreateConditionMutex(
    std::unique_ptr<IConditionMutex>& result, const std::string& name = "") override;
};
