/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/winmutex.h"

#include "twitchsdk/core/assertion.h"

#include <windows.h>

static_assert(sizeof(size_t) == sizeof(HANDLE), "HANDLE and size_t are not the same size");

ttv::WinMutex::WinMutex(const std::string& /*name*/) : mLocked(false) {
  InitializeCriticalSection(&mCriticalSection);
}

ttv::WinMutex::~WinMutex() {
  (void)Destroy();
}

TTV_ErrorCode ttv::WinMutex::Destroy() {
  DeleteCriticalSection(&mCriticalSection);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinMutex::Lock() {
  EnterCriticalSection(&mCriticalSection);

  TTV_ASSERT(!mLocked);
  mLocked = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinMutex::TryLock() {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  BOOL result = TryEnterCriticalSection(&mCriticalSection);
  if (result == 0) {
    ret = TTV_EC_MUTEX_NOT_ACQUIRED;
  } else {
    TTV_ASSERT(!mLocked);
    mLocked = true;
  }

  return ret;
}

TTV_ErrorCode ttv::WinMutex::Unlock() {
  TTV_ASSERT(mLocked);
  mLocked = false;
  LeaveCriticalSection(&mCriticalSection);

  return TTV_EC_SUCCESS;
}

ttv::WinConditionMutex::WinConditionMutex(const std::string& /*name*/) : mLocked(false) {
  InitializeCriticalSection(&mCriticalSection);
  InitializeConditionVariable(&mConditionVariable);
}

ttv::WinConditionMutex::~WinConditionMutex() {
  (void)Destroy();
}

TTV_ErrorCode ttv::WinConditionMutex::Destroy() {
  DeleteCriticalSection(&mCriticalSection);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinConditionMutex::Lock() {
  EnterCriticalSection(&mCriticalSection);

  TTV_ASSERT(!mLocked);
  mLocked = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinConditionMutex::TryLock() {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  BOOL result = TryEnterCriticalSection(&mCriticalSection);
  if (result == 0) {
    ret = TTV_EC_MUTEX_NOT_ACQUIRED;
  } else {
    TTV_ASSERT(!mLocked);
    mLocked = true;
  }

  return ret;
}

TTV_ErrorCode ttv::WinConditionMutex::Unlock() {
  TTV_ASSERT(mLocked);
  mLocked = false;
  LeaveCriticalSection(&mCriticalSection);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinConditionMutex::Wait() {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  TTV_ASSERT(mLocked);
  mLocked = false;
  BOOL result = SleepConditionVariableCS(&mConditionVariable, &mCriticalSection, INFINITE);
  TTV_ASSERT(!mLocked);
  mLocked = true;
  if (result == 0) {
    ret = TTV_EC_CONDITION_WAIT_FAILED;
  }

  return ret;
}

TTV_ErrorCode ttv::WinConditionMutex::WaitFor(uint64_t timeoutMilliseconds) {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  TTV_ASSERT(mLocked);
  mLocked = false;
  BOOL result =
    SleepConditionVariableCS(&mConditionVariable, &mCriticalSection, static_cast<DWORD>(timeoutMilliseconds));
  TTV_ASSERT(!mLocked);
  mLocked = true;
  if (result == 0) {
    DWORD errorCode = GetLastError();

    if (errorCode == ERROR_TIMEOUT) {
      ret = TTV_EC_WAIT_TIMEOUT;
    } else {
      ret = TTV_EC_CONDITION_WAIT_FAILED;
    }
  }

  return ret;
}

TTV_ErrorCode ttv::WinConditionMutex::Signal() {
  WakeConditionVariable(&mConditionVariable);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinConditionMutex::Broadcast() {
  WakeAllConditionVariable(&mConditionVariable);

  return TTV_EC_SUCCESS;
}

#pragma push_macro("CreateMutex")
#undef CreateMutex

TTV_ErrorCode ttv::WinMutexFactory::CreateMutex(std::unique_ptr<ttv::IMutex>& result, const std::string& name) {
  result.reset(new WinMutex(name));

  return TTV_EC_SUCCESS;
}

#pragma pop_macro("CreateMutex")

TTV_ErrorCode ttv::WinMutexFactory::CreateConditionMutex(
  std::unique_ptr<ttv::IConditionMutex>& result, const std::string& name) {
  result.reset(new WinConditionMutex(name));

  return TTV_EC_SUCCESS;
}
