/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/winthread.h"

#include "twitchsdk/core/assertion.h"

#include <windows.h>

#include <thread>

namespace {
#pragma pack(push, 8)
typedef struct tagTHREADNAME_INFO {
  DWORD dwType;      // Must be 0x1000.
  LPCSTR szName;     // Pointer to name (in user addr space).
  DWORD dwThreadID;  // Thread ID (-1=caller thread).
  DWORD dwFlags;     // Reserved for future use, must be zero.
} THREADNAME_INFO;
#pragma pack(pop)

const DWORD MS_VC_EXCEPTION = 0x406D1388;

void SetThreadName(const char* name, uint threadId) {
  THREADNAME_INFO info;
  info.dwType = 0x1000;
  info.szName = name;
  info.dwThreadID = threadId;
  info.dwFlags = 0;

  __try {
    RaiseException(MS_VC_EXCEPTION, 0, sizeof(info) / sizeof(ULONG_PTR), (ULONG_PTR*)&info);
  }
#pragma warning(suppress : 6320)  // intentionally masking an exception here
  __except (EXCEPTION_EXECUTE_HANDLER)
#pragma warning(suppress : 6322)  // intentionally not doing anything if exception thrown
  {
  }
}
}  // namespace

ttv::WinThread::WinThread(ThreadProc threadProc, const std::string& name) : mThreadProc(threadProc), mName(name) {}

ttv::WinThread::~WinThread() {
  // deleting an active thread will crash so make sure it's done
  if (Joinable()) {
    Join();
  }

  mThread.reset();

  ttv::trace::Message("Thread", MessageLevel::Info, "Thread destroyed");
}

void ttv::WinThread::Run() {
  mThread = std::make_unique<std::thread>(std::thread([this] {
    ttv::trace::Message("Thread", MessageLevel::Info, "Thread %s starting", mName.c_str());

    SetThreadName(mName.c_str(), GetCurrentThreadId());

    mThreadProc();

    ttv::trace::Message("Thread", MessageLevel::Info, "Thread %s done", mName.c_str());
  }));

  ttv::trace::Message("Thread", MessageLevel::Info, "Thread created");
}

void ttv::WinThread::Join() {
  TTV_ASSERT(mThread != nullptr);
  TTV_ASSERT(Joinable());

  mThread->join();
}

bool ttv::WinThread::Joinable() const {
  if (mThread == nullptr) {
    return false;
  }

  return mThread->joinable();
}

ttv::WinThreadFactory::~WinThreadFactory() {}

TTV_ErrorCode ttv::WinThreadFactory::CreateThread(
  ThreadProc threadProc, const std::string& name, std::shared_ptr<IThread>& result) {
  result = std::make_shared<WinThread>(threadProc, name);

  return TTV_EC_SUCCESS;
}
