/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/winthreadsync.h"

#include <windows.h>

ttv::WinThreadSync::WinThreadSync() : mSyncAtom(nullptr), mFallthru(false) {
#if WINVER >= _WIN32_WINNT_VISTA
  mSyncAtom = CreateEventEx(nullptr, nullptr, 0, EVENT_ALL_ACCESS);
#else
  mSyncAtom = CreateEvent(0, FALSE, FALSE, 0);
#endif
  assert(mSyncAtom);
}

ttv::WinThreadSync::~WinThreadSync() {
  if (mSyncAtom != nullptr) {
    (void)CloseHandle(mSyncAtom);
  }
}

TTV_ErrorCode ttv::WinThreadSync::Suspend() {
  return SuspendFor(INFINITE);
}

TTV_ErrorCode ttv::WinThreadSync::SuspendFor(uint duration) {
  DWORD waitResult = WaitForSingleObjectEx(mSyncAtom, duration, FALSE);

  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  switch (waitResult) {
    case WAIT_OBJECT_0:
      break;
    case WAIT_TIMEOUT:
      ret = TTV_EC_WAIT_TIMEOUT;
      break;
    default:
      ttv::trace::Message("Thread", MessageLevel::Error, "WaitForSIngleObject returned an error");
      ret = TTV_EC_CANNOT_SUSPEND_THREADSYNC;
  }

  return ret;
}

TTV_ErrorCode ttv::WinThreadSync::Signal() {
  BOOL ret = SetEvent(mSyncAtom);
  if (ret != TRUE) {
    ttv::trace::Message("Thread", MessageLevel::Error, "SetEvent returned an error");
    return TTV_EC_CANNOT_SIGNAL_THREADSYNC;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::WinThreadSyncFactory::CreateThreadSync(
  std::shared_ptr<ttv::IThreadSync>& result, const std::string& name) {
  result = std::make_shared<WinThreadSync>();

  return TTV_EC_SUCCESS;
}
