#pragma once

#include "stringutilities.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <functional>
#include <iostream>
#include <map>
#include <memory>

#if TTV_PLATFORM_DARWIN
#define sscanf_s sscanf
#endif

class CommandDefinition;
class CommandInstance;
class Context;
class IConsoleInputProvider;

typedef std::map<std::string, std::string> VariableMap;

struct ParamType {
  enum Enum { UInt32, Int32, UInt64, Int64, Bool, Float, String, RemainderAsString };
};

class ParamDefinition {
 public:
  ParamDefinition(const std::string& name, ParamType::Enum type);

  ParamType::Enum GetType() const { return mType; }
  std::string GetName() const { return mName; }

 private:
  std::string mName;
  ParamType::Enum mType;
};

class ParamValue {
 public:
  ParamValue(std::shared_ptr<ParamDefinition> def, const std::string& value);
  ParamValue(std::shared_ptr<ParamDefinition> def, uint32_t value);
  ParamValue(std::shared_ptr<ParamDefinition> def, int32_t value);
  ParamValue(std::shared_ptr<ParamDefinition> def, uint64_t value);
  ParamValue(std::shared_ptr<ParamDefinition> def, int64_t value);
  ParamValue(std::shared_ptr<ParamDefinition> def, float value);
  ParamValue(std::shared_ptr<ParamDefinition> def, bool value);

  operator std::string() const;
  operator uint32_t() const;
  operator int32_t() const;
  operator uint64_t() const;
  operator int64_t() const;
  operator float() const;
  operator bool() const;
  bool operator==(const char* other) const;
  bool operator==(const std::string& other) const;
  bool operator==(bool other) const;

  std::string GetString() const;
  uint32_t GetUInt32() const;
  int32_t GetInt32() const;
  uint64_t GetUInt64() const;
  int64_t GetInt64() const;
  float GetFloat() const;
  bool GetBool() const;

 private:
  std::shared_ptr<ParamDefinition> mDefinition;
  std::string mString;
  uint32_t mUInt32;
  int32_t mInt32;
  uint64_t mUInt64;
  int64_t mInt64;
  float mFloat;
  bool mBool;
};

class CommandFunction {
 public:
  using CommandFunc = std::function<bool(const std::vector<ParamValue>& params)>;
  using CommandWithInstanceFunc =
    std::function<bool(const CommandInstance& instance, const std::vector<ParamValue>& params)>;

 public:
  CommandFunction(std::shared_ptr<CommandDefinition> definition);

  CommandFunction& Function(CommandFunc&& func);
  CommandFunction& Function(CommandWithInstanceFunc&& func);
  CommandFunction& Description(const std::string& desc);
  CommandFunction& AddParam(const std::string& name, ParamType::Enum type);
  CommandDefinition& Done();

  const std::vector<std::shared_ptr<ParamDefinition>>& GetParams() const { return mParams; }
  const std::string& GetDescription() const { return mDescription; }

  void SubstituteVariables(std::vector<std::string>& tokens, const std::map<std::string, std::string>& variables) const;
  std::string SubstituteVariables(const std::string& str, const std::map<std::string, std::string>& variables) const;
  bool TryParse(const std::string& paramLine, const std::map<std::string, std::string>& variables,
    std::vector<ParamValue>& values) const;
  bool Invoke(const CommandInstance& instance, const std::vector<ParamValue>& values) const;

 private:
  std::shared_ptr<CommandDefinition> mDefinition;
  std::vector<std::shared_ptr<ParamDefinition>> mParams;
  std::string mDescription;
  CommandFunc mFunction;
  CommandWithInstanceFunc mFunctionWithInstance;
};

class CommandDefinition : public std::enable_shared_from_this<CommandDefinition> {
 public:
  CommandDefinition(const std::string& name);

  std::string GetName() const { return mName; }
  const std::vector<std::string>& GetAliases() const { return mAliases; }
  bool GetExecuteOnMainThread() const { return mExecuteOnMainThread; }
  const std::vector<std::shared_ptr<CommandFunction>>& GetFlavors() const { return mFlavors; }

  CommandFunction& AddFunction();
  CommandDefinition& AddAlias(const std::string& alias);
  CommandDefinition& RunOnMainThread();

  bool IsCommand(const std::string& name) const;

  void PrintUsage() const;

 private:
  std::string mName;
  std::vector<std::shared_ptr<CommandFunction>> mFlavors;
  std::vector<std::string> mAliases;
  bool mExecuteOnMainThread;
};

class CommandInstance {
 public:
  CommandInstance(
    const std::shared_ptr<CommandDefinition>& definition, const std::string& params, const std::string& line);

  void SetEcho(bool echo) { mEcho = echo; }
  bool GetEcho() const { return mEcho; }
  std::string GetLine() const { return mLine; }

  bool GetExecuteOnMainThread() const { return mDefinition->GetExecuteOnMainThread(); }
  std::shared_ptr<CommandDefinition> GetDefinition() { return mDefinition; }

  void SetWorkingDirectory(const std::string& dir) { mWorkingDirectory = dir; }
  const std::string& GetWorkingDirectory() const { return mWorkingDirectory; }

  bool Invoke(VariableMap& variables);
  void PrintUsage() const;

 private:
  std::shared_ptr<CommandDefinition> mDefinition;
  std::string mParams;
  std::vector<std::string> mParamTokens;
  std::string mLine;
  std::string mWorkingDirectory;
  bool mEcho;
};

class CommandCategory {
 public:
  CommandCategory(const std::string& name, const std::string& description);

  std::string GetName() const { return mName; }
  const std::vector<std::shared_ptr<CommandDefinition>>& GetCommands() const { return mDefinitions; }

  CommandDefinition& AddCommand(const std::string& name);

  bool FindCommand(const std::string& name, std::shared_ptr<CommandDefinition>& result);

 private:
  std::string mName;
  std::string mDescription;
  std::vector<std::shared_ptr<CommandDefinition>> mDefinitions;
};

/**
 * App state which can be manipulated by cli modules.
 */
class Context {
 public:
  Context();

  void SetConsoleInputProvider(const std::shared_ptr<IConsoleInputProvider>& input) { mConsoleInputProvider = input; }
  std::shared_ptr<IConsoleInputProvider> GetConsoleInputProvider() const { return mConsoleInputProvider; }

  void RegisterCommandCategory(std::shared_ptr<CommandCategory> category);

  void RegisterModule(std::shared_ptr<ttv::IModule> module);
  std::shared_ptr<ttv::IModule> GetModule(const std::string& name);

  void RegisterLoggers(const char* const* loggers, size_t numLoggers);
  const std::vector<std::string>& GetLoggers() const { return mLoggers; }

  void InitializeModules();
  void UpdateModules();
  void ShutdownModules();

  void Exit();
  bool ShouldExit() const { return mExit; }

  void SetUpdateInterval(uint64_t interval) { mUpdateInterval = interval; }
  uint64_t GetUpdateInterval() const { return mUpdateInterval; }

  void SetVariable(const std::string& name, const std::string& value);
  const std::map<std::string, std::string>& GetVariables() const { return mVariables; }

  bool GenerateCommand(const std::string& line, std::shared_ptr<CommandInstance>& result);
  bool EnqueueCommand(const std::string& line, const std::string& workingDirectory, bool echo);
  bool SetCommandAlias(const std::string& cmd, const std::string& alias);

  void PrintHelpCategory(const std::string& categoryName);
  void PrintHelp();

  bool ProcessCommand();

 private:
  std::shared_ptr<IConsoleInputProvider> mConsoleInputProvider;
  std::vector<std::shared_ptr<ttv::IModule>> mModules;
  std::map<std::string, std::shared_ptr<CommandCategory>> mCommands;
  VariableMap mVariables;
  ttv::ConcurrentQueue<std::shared_ptr<CommandInstance>> mCommandQueue;
  std::vector<std::string> mLoggers;
  uint64_t mUpdateInterval;
  bool mExit;
};

void ReportCommandResult(TTV_ErrorCode ec);
