#include "context.h"
#include "staticinit.h"
#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/tracer.h"

#include <regex>
#include <sstream>

namespace {
std::shared_ptr<CommandCategory> RegisterShellCommands(std::shared_ptr<Context> context) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Shell", "General commands related to script management");

  category->AddCommand("Help")
    .AddAlias("?")
    .AddFunction()
    .Description("Displays all help")
    .Function([context](const std::vector<ParamValue> & /*params*/) -> bool {
      context->PrintHelp();

      return true;
    })
    .Done()
    .AddFunction()
    .Description("Displays help for the given category")
    .AddParam("category", ParamType::String)
    .Function([context](const std::vector<ParamValue>& params) -> bool {
      context->PrintHelpCategory(params[0]);

      return true;
    });

  category->AddCommand("Tips")
    .RunOnMainThread()
    .AddFunction()
    .Description("Shows some helpful tips")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::cout << std::endl;
      std::cout << "Tips" << std::endl;
      std::cout << std::endl;
      std::cout << "  * Use the \"run\" command to run script files." << std::endl;
      std::cout
        << "  * Automatically run a script file on launch by passing the path as the first command line argument to chatshell."
        << std::endl;
      std::cout << "  * Set variables for usernames, oauth tokens and anything else using the \"set\" command."
                << std::endl;
      std::cout << "  * Command names are case insensitive." << std::endl;
      std::cout << "  * Make sure you \"sleep\" a bit when waiting for a callback for an async operation." << std::endl;
      std::cout << std::endl;

      return true;
    });

  category->AddCommand("SetVariable")
    .AddAlias("set")
    .AddFunction()
    .Description("Sets a variable value.  Use the variable by preceeding with $")
    .AddParam("name", ParamType::String)
    .AddParam("value", ParamType::RemainderAsString)
    .Function([context](const std::vector<ParamValue>& params) -> bool {
      context->SetVariable(params[0], params[1].GetString());
      std::cout << "  " << params[0].GetString() << " = " << params[1].GetString() << std::endl;
      return true;
    });

  category->AddCommand("Trace")
    .AddFunction()
    .Description("Turns logging on and off for the named logger.")
    .AddParam("on|off", ParamType::String)
    .AddParam("logger", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      ttv::MessageLevel level = ttv::MessageLevel::None;

      if (params[0] == "on") {
        level = ttv::MessageLevel::Debug;
      }

      ttv::trace::SetComponentMessageLevel(params[1].GetString().c_str(), level);

      return true;
    })
    .Done()
    .AddFunction()
    .Description("Turns logging on or off for most loggers.")
    .AddParam("on|off", ParamType::String)
    .Function([context](const std::vector<ParamValue>& params) -> bool {
      ttv::MessageLevel level = ttv::MessageLevel::None;

      if (params[0] == "on") {
        level = ttv::MessageLevel::Debug;
      } else if (params[0] == "off") {
        level = ttv::MessageLevel::None;
      } else {
        return false;
      }

      const auto& loggers = context->GetLoggers();
      for (const auto& logger : loggers) {
        ttv::trace::SetComponentMessageLevel(logger.c_str(), level);
      }

      return true;
    });

  category->AddCommand("RunScript")
    .AddAlias("run")
    .RunOnMainThread()
    .AddFunction()
    .Description("Runs a script file")
    .AddParam("path", ParamType::String)
    .Function([context](const CommandInstance& instance, const std::vector<ParamValue>& params) -> bool {
      std::string path = params[0];

      bool loaded = false;
      std::vector<std::string> lines;
      loaded = ttv::ReadFileLines(path, lines);
      if (!loaded) {
        if (!instance.GetWorkingDirectory().empty()) {
          path = instance.GetWorkingDirectory() + "/" + path;
          loaded = ttv::ReadFileLines(path, lines);
        }
      }

      if (!loaded) {
        context->EnqueueCommand("echo Could not read script file", "", false);
        return true;
      }

      context->EnqueueCommand("echo Running script file " + path + "...", "", false);

      // Extract the working directory
      std::replace(path.begin(), path.end(), '\\', '/');

      size_t index = path.rfind('/');
      if (index == std::string::npos) {
        path.clear();
      } else {
        path = path.substr(0, index);
      }

      // Add each line of the file as a command
      bool inBlockComment = false;
      for (auto line : lines) {
        if (line == "" || line[0] == '#') {
          continue;
        } else if (line == "/*") {
          inBlockComment = true;
        } else if (line == "*/") {
          inBlockComment = false;
        } else if (!inBlockComment) {
          context->EnqueueCommand(line, path, true);
        }
      }

      context->EnqueueCommand("echo Done running script file", "", false);

      return true;
    });

  category->AddCommand("Sleep")
    .RunOnMainThread()
    .AddFunction()
    .Description("Pauses before processing the next command")
    .AddParam("milliseconds", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      ttv::Sleep(static_cast<uint>(params[0].GetUInt32()));

      return true;
    });

  category->AddCommand("Quit")
    .AddAlias("exit")
    .AddFunction()
    .Description("Exits the chat shell")
    .Function([context](const std::vector<ParamValue> & /*params*/) -> bool {
      context->Exit();
      return true;
    });

  category->AddCommand("Alias")
    .AddFunction()
    .Description("Creates a new alias for a command")
    .AddParam("alias", ParamType::String)
    .AddParam("command", ParamType::String)
    .Function([context](const std::vector<ParamValue>& params) -> bool {
      if (!context->SetCommandAlias(params[0], params[1])) {
        std::cout << "Unknown command, could not create alias" << std::endl;
      }

      return true;
    });

  category->AddCommand("Echo")
    .AddAlias("print")
    .AddFunction()
    .Description("Echoes the given message")
    .AddParam("message", ParamType::RemainderAsString)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::cout << params[0].GetString() << std::endl;
      return true;
    });

  category->AddCommand("SetUpdateInterval")
    .AddFunction()
    .Description("Sets the number of milliseconds to sleep between module Update() calls.")
    .AddParam("miliseconds", ParamType::UInt64)
    .Function([context](const std::vector<ParamValue>& params) -> bool {
      context->SetUpdateInterval(params[0]);
      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      auto commands = RegisterShellCommands(context);
      context->RegisterCommandCategory(commands);
    },
    999);
});
}  // namespace
