#include "main.h"

#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/threadsync.h"
#include "twitchsdk/core/tracer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <assert.h>

#include <iostream>
#include <map>
#include <memory>
#include <string>

namespace {
using namespace ttv;

std::shared_ptr<Context> gContext;
}  // namespace

#if TTV_TARGET_IOS

bool IosConsoleInputProvider::GetLine(std::string &line) {
  line = "";

  volatile int blah = 0;

  // TODO: Implement a dialog box on iOS to allow input
  while (blah == 0) {
    ttv::Sleep(100);
  }

  return false;
}

#else

bool StdInConsoleInputProvider::GetLine(std::string& line) {
  line = "";

  const auto& stream = std::getline(std::cin, line);

  if (stream.fail()) {
    if (stream.bad()) {
      std::cout << "Failed to read from stdin: " << strerror(errno) << std::endl << "Exiting." << std::endl;
      gContext->Exit();
    } else {
      std::cout << "Received non-fatal error while reading from stdin: " << strerror(errno) << std::endl
                << "Clearing and retrying." << std::endl;
      std::cin.clear();
    }
  }

  return line != "";
}

#endif

int run(int argc, char *argv[]) {
  TTV_ErrorCode ec = TTV_InitializeLibrary();

  gContext = std::make_shared<Context>();

#if TTV_TARGET_IOS
  gContext->SetConsoleInputProvider(std::make_shared<IosConsoleInputProvider>());
#else
  gContext->SetConsoleInputProvider(std::make_shared<StdInConsoleInputProvider>());
#endif

  if (TTV_SUCCEEDED(ec)) {
    // Load all the modules
    StaticInitializationRegistrar::GetInstance().RunInitializers(gContext);

    std::shared_ptr<IThread> thread;
    TTV_ErrorCode ec = ttv::CreateThread(
      []() {
        while (!gContext->ShouldExit()) {
          std::shared_ptr<CommandInstance> cmd;
          while (gContext->ProcessCommand()) {
          }

          gContext->UpdateModules();

          ttv::Sleep(gContext->GetUpdateInterval());
        }
      },
      "Client command thread", thread);

    assert(TTV_SUCCEEDED(ec) && thread != nullptr);

    thread->Run();

    if (argc > 1) {
      gContext->EnqueueCommand(std::string("run ") + Escape(argv[1]), "", false);
    } else {
      std::cout << "Welcome to the definitive tool for proving Drew needs to fix his code!" << std::endl;
      std::cout << "Type \"Help\" for a full list of commands." << std::endl;
      std::cout << "Type \"Tips\" for ways to make this shell easier to use." << std::endl;
      std::cout << std::endl;
    }

    std::string line;
    while (!gContext->ShouldExit()) {
      if (gContext->GetConsoleInputProvider()->GetLine(line)) {
        Trim(line);

        if (line != "") {
          gContext->EnqueueCommand(line, "", false);
        }
      }
    }

    thread->Join();

    gContext->ShutdownModules();

    ec = TTV_ShutdownLibrary();

    if (TTV_FAILED(ec)) {
      std::cout << "Error shutting down Twitch SDK library: " << ErrorToString(ec) << std::endl;
    }
  } else {
    std::cout << "Error initializing Twitch SDK library: " << ErrorToString(ec) << std::endl;
  }

  return static_cast<int>(ec);
}

int main(int argc, char *argv[]) {
  return run(argc, argv);
}
