#include "stringutilities.h"

#include <algorithm>
#include <ctime>
#include <fstream>
#include <functional>
#include <iostream>
#include <sstream>

#if TTV_TARGET_IOS
#include <CoreFoundation/CoreFoundation.h>
#endif

using namespace ttv;

std::string ttv::TrimStringQuotes(const std::string& str) {
  // Trim the quotes from around the key
  if (str.size() >= 2 && str[0] == '"' && str[str.size() - 1] == '"') {
    return str.substr(1, str.size() - 2);
  } else {
    return str;
  }
}

void ttv::Split(const std::string& str, std::vector<std::string>& out, char sep) {
  unsigned start = 0;
  unsigned end = 0;

  for (;;) {
    if (end == str.size() || str[end] == sep) {
      if (end > start) {
        out.push_back(str.substr(start, end - start));
      }

      if (end == str.size()) {
        break;
      }

      ++end;
      start = end;
    } else {
      ++end;
    }
  }
}

std::string ttv::Escape(const std::string& str) {
  std::string result;

  for (size_t i = 0; i < str.size(); ++i) {
    if (str[i] == '\\') {
      result += '\\';
    }

    result += str[i];
  }

  return result;
}

void ttv::ToLowerCase(std::string& str) {
  (void)std::transform(str.begin(), str.end(), str.begin(), ::tolower);
}

bool ttv::ParseCommand(const std::string& line, std::string& command, std::string& params) {
  std::string str = line;
  Trim(str);

  if (str == "") {
    return false;
  }

  size_t index = str.find(" ");
  if (index != std::string::npos) {
    command = str.substr(0, index);
    params = str.substr(index + 1);
  } else {
    command = str;
    params = "";
  }

  return true;
}

void ttv::TokenizeParameters(const std::string& str, std::vector<std::string>& result) {
  std::string raw = str;
  Trim(raw);

  bool inQuote = false;
  bool escape = false;

  std::string token;

  auto endOfToken = [&result, &token](bool allowEmpty) {
    if (allowEmpty || !token.empty()) {
      result.push_back(token);
      token.clear();
    }
  };

  for (size_t i = 0; i < raw.size(); ++i) {
    char ch = raw[i];

    if (escape) {
      escape = false;

      switch (ch) {
        case 't':
          token += '\t';
          break;
        case 'n':
          token += '\n';
          break;
        case 'r':
          token += '\r';
          break;
        case '"':
          break;
        default:
          token += ch;
          break;
      }
    } else if (ch == '\\') {
      escape = true;
    } else if ((ch == ' ' || ch == '\t') && !inQuote) {
      endOfToken(false);
    } else if (ch == '"') {
      if (inQuote) {
        endOfToken(true);
      } else {
        endOfToken(false);
      }

      inQuote = !inQuote;
    } else {
      token += ch;
    }
  }

  endOfToken(false);
}

const char* ttv::AdvanceToNextWord(const char* str) {
  if (str == nullptr || *str == '\0') {
    return nullptr;
  }

  // Skip word characters
  if (!IsWhitespace(*str)) {
    while (*str != '\0' && !IsWhitespace(*str)) {
      str++;
    }
  }

  // Skip whitespace
  while (*str != '\0' && IsWhitespace(*str)) {
    str++;
  }

  return str;
}

std::string ttv::ToString(Timestamp timestamp) {
  time_t t(timestamp);
  std::string str(ctime(&t));
  Trim(str);
  return str;
}

std::string ttv::ToString(ttv::IModule::State state) {
  switch (state) {
    case IModule::State::Uninitialized:
      return "Uninitialized";
    case IModule::State::Initializing:
      return "Initializing";
    case IModule::State::Initialized:
      return "Initialized";
    case IModule::State::ShuttingDown:
      return "ShuttingDown";
    default:
      return "?????";
  }
}

std::string ttv::ToString(ttv::VodType type) {
  switch (type) {
    case VodType::Archive:
      return "Archive";
    case VodType::Highlight:
      return "Highlight";
    case VodType::Upload:
      return "Upload";
    case VodType::Unknown:
      return "Unknown";
    default:
      return "?????";
  }
}

std::string ttv::ToString(ttv::BroadcastPlatform type) {
  switch (type) {
    case BroadcastPlatform::WatchParty:
      return "WatchParty";
    case BroadcastPlatform::Premiere:
      return "Premiere";
    case BroadcastPlatform::Rerun:
      return "Rerun";
    case BroadcastPlatform::Playlist:
      return "Playlist";
    case BroadcastPlatform::Mobile:
      return "Mobile";
    case BroadcastPlatform::Xbox:
      return "Xbox";
    case BroadcastPlatform::PS4:
      return "PS4";
    case BroadcastPlatform::Live:
      return "Live";
    case BroadcastPlatform::Unknown:
      return "Unknown";
    default:
      return "?????";
  }
}

std::string ttv::ToString(ttv::StreamType type) {
  switch (type) {
    case StreamType::WatchParty:
      return "WatchParty";
    case StreamType::Premiere:
      return "Premiere";
    case StreamType::Rerun:
      return "Rerun";
    case StreamType::Playlist:
      return "Playlist";
    case StreamType::Live:
      return "Live";
    case StreamType::Unknown:
      return "Unknown";
    default:
      return "?????";
  }
}

void ttv::Print(const UserInfo& userInfo, std::string indent) {
  std::cout << indent << "UserInfo: " << userInfo.userName << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(userInfo, displayName) << std::endl;
  std::cout << indent << PRINT_FIELD(userInfo, userId) << std::endl;
  std::cout << indent << PRINT_FIELD(userInfo, logoImageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(userInfo, bio) << std::endl;
}

void ttv::Print(const std::vector<UserInfo>& userInfoList, std::string indent) {
  std::cout << indent << "UserInfoList[" << userInfoList.size() << "]:" << std::endl;

  for (const auto& userInfo : userInfoList) {
    Print(userInfo, indent + "  ");
  }
}

void ttv::Print(const ttv::ChannelInfo& channelInfo, std::string indent) {
  std::cout << indent << "ChannelInfo: " << channelInfo.channelId << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(channelInfo, channelId) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, displayName) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, name) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, game) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, description) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, status) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, language) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, broadcasterLanguage) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, logoImageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, channelUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, videoBannerImageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, profileBannerImageUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, createdAtTimestamp) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, updatedAtTimestamp) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, numFollowers) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, numViews) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, mature) << std::endl;
  std::cout << indent << PRINT_FIELD(channelInfo, partner) << std::endl;
}

void ttv::Print(const ttv::PreviewImages& images, std::string indent) {
  std::cout << indent << "PreviewImages:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(images, largeUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(images, mediumUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(images, smallUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(images, templateUrl) << std::endl;
}

void ttv::Print(const ttv::StreamInfo& streamInfo, std::string indent) {
  std::cout << indent << "StreamInfo: " << streamInfo.streamId << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(streamInfo, streamId) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, game) << std::endl;
  Print(streamInfo.previewImages, indent);
  std::cout << indent << PRINT_FIELD(streamInfo, averageFPS) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, delay) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, viewerCount) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, videoHeight) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, createdAtTimestamp) << std::endl;
  std::cout << indent << "broadcastPlatform: " << ToString(streamInfo.broadcastPlatform) << std::endl;
  std::cout << indent << "streamType: " << ToString(streamInfo.streamType) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfo, isPlaylist) << std::endl;

  Print(streamInfo.channelInfo, indent);
}

void ttv::Print(const ttv::StreamInfoUpdate& streamInfoUpdate, std::string indent) {
  std::cout << indent << "StreamInfoUpdate: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(streamInfoUpdate, title) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfoUpdate, game) << std::endl;
  std::cout << indent << PRINT_FIELD(streamInfoUpdate, gameId) << std::endl;
}

void ttv::Print(const ttv::WatchPartyUpdate& update, std::string indent) {
  std::cout << indent << "WatchPartyUpdate:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(update, incrementUrl) << std::endl;
  std::cout << indent << PRINT_FIELD(update, vodId) << std::endl;
  std::cout << indent << PRINT_FIELD(update, title) << std::endl;
  std::cout << indent << PRINT_FIELD(update, watchPartyId) << std::endl;
  std::cout << indent << "broadcastType: " << ToString(update.broadcastType);
  std::cout << indent << PRINT_FIELD(update, viewable) << std::endl;
}

void ttv::Print(const ttv::ProfileImage& image, std::string indent) {
  std::cout << indent << "ProfileImage:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(image, width) << std::endl;
  std::cout << indent << PRINT_FIELD(image, height) << std::endl;
  std::cout << indent << PRINT_FIELD(image, url) << std::endl;
  std::cout << indent << PRINT_FIELD(image, format) << std::endl;
}

void ttv::Print(const SquadStatus& squad, std::string indent) {
  std::cout << indent << "SquadStatus: ";

  switch (squad) {
    case SquadStatus::Unknown:
      std::cout << "unknown";
      break;
    case SquadStatus::Pending:
      std::cout << "pending";
      break;
    case SquadStatus::Live:
      std::cout << "live";
      break;
    case SquadStatus::Ended:
      std::cout << "ended";
      break;
  }

  std::cout << std::endl;
}

void ttv::Print(const SquadMember& squadMember, std::string indent) {
  std::cout << indent << "SquadMember: " << std::to_string(squadMember.channelId) << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(squadMember, userLogin) << std::endl;
  std::cout << indent << PRINT_FIELD(squadMember, userDisplayName) << std::endl;
  std::cout << indent << PRINT_FIELD(squadMember, profileImageUrl150) << std::endl;
}

void ttv::Print(const SquadInfo& squad, std::string indent) {
  std::cout << indent << "SquadInfo:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(squad, squadId) << std::endl;
  std::cout << indent << PRINT_FIELD(squad, ownerId) << std::endl;

  Print(squad.status, indent);

  std::cout << indent << "SquadMembers:" << std::endl;
  for (const auto& member : squad.members) {
    Print(member, indent + "  ");
  }
}

void ttv::Print(const CanTheyError& error, std::string indent) {
  std::cout << indent << "CanTheyError:" << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_FIELD(error, code) << std::endl;
  std::cout << indent << PRINT_FIELD(error, message) << std::endl;
  for (const auto& link : error.links) {
    std::cout << indent << PRINT_VARIABLE(link) << std::endl;
  }
}

#if TTV_TARGET_IOS
#include <CoreFoundation/CoreFoundation.h>
#endif

bool ttv::ReadFileLines(const std::string& path, std::vector<std::string>& lines) {
  std::string loadPath = path;

#if TTV_TARGET_IOS
  CFBundleRef bundle = CFBundleGetMainBundle();

  if (bundle == nullptr) {
    return false;
  }

  CFStringRef refPath = CFStringCreateWithCString(kCFAllocatorDefault, path.c_str(), kCFStringEncodingUTF8);
  CFURLRef url = CFBundleCopyResourceURL(bundle, refPath, nullptr, nullptr);
  CFStringRef str = CFURLGetString(url);
  loadPath = CFStringGetCStringPtr(str, kCFStringEncodingUTF8);

  const std::string prefix = "file://";
  if (ttv::StartsWith(loadPath, prefix)) {
    loadPath = loadPath.substr(prefix.size());
  }

  CFRelease(str);
  CFRelease(refPath);
#endif

  std::ifstream file(loadPath);

  // file doesn't exist
  if (!file.is_open()) {
    return false;
  }

  while (!file.eof()) {
    std::string line;
    std::getline(file, line);
    lines.push_back(line);
  }

  file.close();

#if TTV_TARGET_IOS
  CFRelease(bundle);
#endif

  return true;
}

std::string ttv::CommaSeparatedStrings(
  const std::vector<std::pair<bool, std::string>>& optionalStrings, const std::string& emptyPlaceholder) {
  bool empty = true;
  std::stringstream stream;
  for (const auto& pair : optionalStrings) {
    if (pair.first) {
      if (empty) {
        empty = false;
      } else {
        stream << ", ";
      }

      stream << pair.second;
    }
  }

  return empty ? emptyPlaceholder : stream.str();
}

std::string ttv::ContainerizedPath(const std::string& path) {
#if TTV_TARGET_IOS
  CFStringRef pathString = CFStringCreateWithCString(kCFAllocatorDefault, path.c_str(), kCFStringEncodingUTF8);
  CFURLRef homeDirectoryUrl = CFCopyHomeDirectoryURL();
  CFURLRef containerizedUrl = CFURLCreateCopyAppendingPathComponent(NULL, homeDirectoryUrl, pathString, false);
  CFStringRef containerizedPath = CFURLCopyFileSystemPath(containerizedUrl, kCFURLPOSIXPathStyle);
  std::string cppContainerizedPath = CFStringGetCStringPtr(containerizedPath, kCFStringEncodingUTF8);

  CFRelease(pathString);
  CFRelease(homeDirectoryUrl);
  CFRelease(containerizedUrl);
  CFRelease(containerizedPath);

  return cppContainerizedPath;
#else
  return path;
#endif
}
