package tv.twitch.test;

import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;

import tv.twitch.*;

public abstract class TestBase {
    // For testing purposes - clear these values before you check in
    protected static String m_OauthToken = "";
    protected static int m_UserId = -1;

    protected Library m_Library = null;
    protected boolean m_Passed = true;
    protected ArrayList<IModule> m_Modules = new ArrayList<>();

    protected static byte[] loadFileAsBytes(String dir, String filename) {
        try {
            Path file = Paths.get(dir, filename);
            return Files.readAllBytes(file);
        } catch (IOException e) {
            return null;
        }
    }

    protected final IModule mDummyModule = new IModule() {
        @Override
        public ErrorCode update() {
            return null;
        }

        @Override
        public ErrorCode shutdown(ShutdownCallback callback) {
            return null;
        }

        @Override
        public ErrorCode initialize(InitializeCallback callback) {
            return null;
        }

        @Override
        public ModuleState getState() {
            return null;
        }

        @Override
        public String getModuleName() {
            return null;
        }
    };

    protected TestBase(Library library) { m_Library = library; }

    public abstract void run() throws Exception;

    public boolean getPassed() { return m_Passed; }

    public String getCurrentMethodName() {
        StackTraceElement[] stack = Thread.currentThread().getStackTrace();
        return stack[2].getMethodName();
    }

    public List<String> getAllMethodNames(Class<?> klass) {
        List<String> result = new ArrayList<>();

        try {
            Method[] method = klass.getDeclaredMethods();
            for (int i = 0; i < method.length; i++) {
                result.add(method[i].getName());
            }
        } catch (Throwable e) {
            return null;
        }

        return result;
    }

    public void checkAllMethodsCalled(Class<?> type, HashSet<String> calls) throws Exception {
        List<String> methodNames = getAllMethodNames(type);
        for (String method : methodNames) {
            if (!calls.contains(method)) {
                throw new Exception(type.getName() + "::" + method + " not called");
            }
        }
    }

    public void checkError(ErrorCode error) throws Exception {
        if (error.failed()) {
            throw new Exception("Received error:" + error.getName());
        }
    }

    protected void initializeModule(IModule module) {
        module.initialize(null);

        while (module.getState() != ModuleState.Initialized) {
            module.update();
            updateModules();

            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }
        }
    }

    protected void shutdownModule(IModule module) {
        module.shutdown(null);

        while (module.getState() != ModuleState.Uninitialized) {
            module.update();
            updateModules();

            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }
        }
    }

    protected void addModule(IModule module) { m_Modules.add(module); }

    protected void removeModule(IModule module) { m_Modules.remove(module); }

    protected void clearModules() { m_Modules.clear(); }

    protected void updateModules() {
        for (IModule module : m_Modules) {
            module.update();
        }
    }

    protected void updateModules(long ms) {
        long num = ms / 100;

        for (long i = 0; i < num; ++i) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }

            updateModules();
        }
    }

    public <T> void ExpectEqual(T expected, T actual) throws Exception {
        if (!expected.equals(actual)) {
            throw new Exception("ExpecteEqual Fails\n" + expected.toString() + "\n" + actual.toString());
        }
    }
}
