#!/usr/bin/python
#
# This tests script will generate projects with various combinations of features
# and ensure they build error free as expected.
#

# Import standard modules
import sys
import os
import argparse
import imp # Used to import the module and fragment files dynamically
import shutil
import re
import git

# Cache the root directory of the test script
test_dir = os.path.realpath(os.path.dirname(__file__))

# Load the generate script
generate_twitchsdk = imp.load_source( 'generate_twitchsdk', os.path.realpath((os.path.join(test_dir, '..', '..', 'build', 'generate_twitchsdk.py'))) )


class Options:

    def __init__(self):

        self.platform = []
        self.targets = []
        self.compilers = []
        self.architectures = []
        self.modules = []
        self.library_types = ['static', 'dynamic']
        self.languages = [None] # The languages to test
        self.settings = []
        self.fragment_search_paths = [] # The directories to search for source fragments
        self.output_dir = None


def parse_command_line(options):

    parser = argparse.ArgumentParser(description='Runs the generation tests.')

    parser.add_argument(
        '--module',
        required=True,
        metavar='<module>',
        action='append',
        help='Adds a module to test.'
    )

    parser.add_argument(
        '--platform',
        required=True,
        metavar='<platform>',
        help='Specifies the platform to test.'
    )

    parser.add_argument(
        '--output-dir',
        required=True,
        metavar='<output_dir>',
        help='Specifies the root output directory for the final project files.'
    )

    parser.add_argument(
        '--fragments-dir',
        required=False,
        metavar='<fragments_path>',
        action='append',
        help='Adds a fragment root directory to test'
    )

    parser.add_argument(
        '--compiler',
        required=True,
        metavar='<compiler>',
        action='append',
        help='Adds a compiler to test.'
    )

    parser.add_argument(
        '--architecture',
        '--arch',
        required=True,
        metavar='<architecture>',
        action='append',
        help='Adds an architecture to test.'
    )

    parser.add_argument(
        '--language',
        required=False,
        metavar='<language>',
        action='append',
        help='Adds a language to test.'
    )

    parser.add_argument(
        '-settings',
        nargs='*',
        action='append',
        help='Settings to pass through'
    )

    args = parser.parse_args()

    options.platform = args.platform
    options.output_dir = args.output_dir

    if not args.fragments_dir is None:
        for x in args.fragments_dir:
            x = os.path.realpath(x)
            options.fragment_search_paths.append(x)

    if not args.module is None:
        for x in args.module:
            options.modules.append(x)

    if not args.architecture is None:
        for x in args.architecture:
            options.architectures.append(x)

    if not args.compiler is None:
        for x in args.compiler:
            options.compilers.append(x)

    if not args.language is None:
        for x in args.language:
            options.languages.append(x)

    if not args.settings is None:
        for arr in args.settings:
            for a in arr:
                options.settings.append(a)


if __name__ == "__main__":

    # Setup options
    options = Options()
    options.fragment_search_paths.append( os.path.realpath(os.path.join(test_dir, '..', '..')) )

    # Determine what we need to do
    parse_command_line(options)

    # Create projects with all the combinations of settings
    def test_configuration(options, args, name):
        # Determine the output directory for the project
        output_dir = os.path.join(options.output_dir, name)

        args = args[:]
        args.append('--output-dir=' + output_dir)
        args.append('--project-name=library')

        # Generate the configuration
        generate_twitchsdk.process(args)

    def test_language(options, args, language, name):
        args = args[:]
        if not language is None:
            args.append('--language=' + language)
            name = name + '_' + language

        test_configuration(options, args, name)

    def test_architecture(options, args, arch, name):
        args = args[:]
        args.append('--arch=' + arch)

        name = name + '_' + arch

        for language in options.languages:
            test_language(options, args, language, name)

    def test_library_type(options, args, library_type, name):
        args = args[:]
        args.append('--' + library_type)

        name = name + '_' + library_type

        for arch in options.architectures:
            test_architecture(options, args, arch, name)

    def test_compiler(options, args, compiler, name):
        args = args[:]
        args.append('--compiler=' + compiler)

        name = name + '_' + compiler

        for library_type in options.library_types:
            test_library_type(options, args, library_type, name)

    def test(options):
        name = options.platform
        args = ['--platform=' + options.platform]

        for module in options.modules:
            args.append('--module=' + module)
            name = name + '_' + module

        for path in options.fragment_search_paths:
            args.append('--fragments-dir=' + path)

        if len(options.settings) > 0:
            args.append('-settings')
            for setting in options.settings:
                args.append(setting)

        for compiler in options.compilers:
            test_compiler(options, args, compiler, name)

    # Test each combination
    test(options)

