/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "fixtures/sdkbasetest.h"
#include "testhttprequest.h"
#include "testutilities.h"
#include "twitchsdk/core/task/simplejsonhttptask.h"
#include "twitchsdk/core/task/taskrunner.h"

#include "gtest/gtest.h"

namespace ttv {
namespace test {
class TaskTest;
}
}  // namespace ttv

class ttv::test::TaskTest : public SdkBaseTest {
 protected:
  TaskTest();
  virtual ~TaskTest();

  virtual void Update() override;

  // Sends a request of TaskType and invokes the passed-in callback when the task finishes.
  // It also asserts that the callback is called with the expected parameters.
  // The callback should satisfy the three-parameter concept laid out by tasks such as GetUserTask.
  template <typename TaskType, typename ErrorType = TTV_ErrorCode>
  void TestSimpleRequest(UserId userId, typename TaskType::Callback callback) {
    bool callbackCalled = false;
    std::shared_ptr<TaskType> task;

    typename TaskType::Callback callbackFunc = [&callbackCalled, &task, callback](TaskType* source, ErrorType error,
                                                 std::shared_ptr<typename TaskType::Result> result) {
      callbackCalled = true;
      ASSERT_EQ(task.get(), source);
      callback(source, error, result);
    };

    auto checkFunction = [&callbackCalled]() { return callbackCalled; };

    auto pollFunction = [this]() { Update(); };

    task = std::make_shared<TaskType>(userId, callbackFunc);
    ASSERT_TRUE(mTaskRunner->AddTask(task));

    WaitUntilResultWithPollTask(1000, checkFunction, pollFunction);

    ASSERT_TRUE(callbackCalled);
  }

  template <typename TaskType, typename... Args>
  std::shared_ptr<typename TaskType::Result> RunTaskWithResult(Args&&... args) {
    std::shared_ptr<TaskType> task;
    std::shared_ptr<typename TaskType::Result> data;
    bool callbackCalled = false;

    typename TaskType::Callback callbackFunc = [&callbackCalled, &task, &data](TaskType* source, TTV_ErrorCode ec,
                                                 std::shared_ptr<typename TaskType::Result> result) {
      callbackCalled = true;
      EXPECT_EQ(task.get(), source);
      data = result;
    };

    auto checkFunction = [&callbackCalled]() { return callbackCalled; };

    auto pollFunction = [this]() { Update(); };

    task = std::make_shared<TaskType>(std::forward<Args>(args)..., callbackFunc);
    EXPECT_TRUE(mTaskRunner->AddTask(task));

    WaitUntilResultWithPollTask(1000, checkFunction, pollFunction);

    EXPECT_TRUE(callbackCalled);

    return data;
  }

  void TestSimpleJsonHttpRequest(const std::string& url, const HttpRequestType method, const std::string& oauthToken,
    SimpleJsonHttpTask::Callback callback) {
    bool callbackCalled = false;
    std::shared_ptr<SimpleJsonHttpTask> task;

    SimpleJsonHttpTask::Callback callbackFunc = [&callbackCalled, &task, callback](SimpleJsonHttpTask* source,
                                                  TTV_ErrorCode ec,
                                                  std::shared_ptr<SimpleJsonHttpTask::Result> result) {
      callbackCalled = true;
      ASSERT_EQ(task.get(), source);
      callback(source, ec, result);
    };

    auto checkFunction = [&callbackCalled]() { return callbackCalled; };

    auto pollFunction = [this]() { Update(); };

    task = std::make_shared<SimpleJsonHttpTask>(url, method, oauthToken, callbackFunc);
    ASSERT_TRUE(mTaskRunner->AddTask(task));

    WaitUntilResultWithPollTask(1000, checkFunction, pollFunction);

    ASSERT_TRUE(callbackCalled);
  }

 protected:
  std::shared_ptr<ttv::TaskRunner> mTaskRunner;
};
