/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/eventqueue.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <atomic>

namespace ttv {
namespace test {
class TestEventScheduler;
}
}  // namespace ttv

/**
 * An implementation of IEventScheduler that is used for test purposes and allows for waiting on events with a timeout.
 *
 * Unlike other IEventScheduler implementations, TestEventScheduler::Shutdown() is synchronous.
 * Shutdown() should be called before the TestEventScheduler is destroyed.
 *
 * ScheduleTask() and CancelTask() should not be called after TestEventScheduler is shut down. Otherwise, it is expected
 * that TTV_EC_NOT_INITIALIZED is returned, but it is not guaranteed.
 */
class ttv::test::TestEventScheduler : public ttv::IEventScheduler {
 public:
  TestEventScheduler();
  TestEventScheduler(const std::string& name);
  virtual ~TestEventScheduler();

  // IEventScheduler overrides
  virtual Result<TaskId> ScheduleTask(TaskParams&& taskParams) override;
  virtual TTV_ErrorCode CancelTask(TaskId taskId) override;
  virtual TTV_ErrorCode Shutdown(TaskFunc&& shutdownTask) override;
  virtual EventSchedulerState GetState() override;

  /**
   * Wait (only for a given timeout) until a task in the TestEventScheduler can be run, remove that task and then run it
   * in the current execution context.
   *
   * WaitForEventWithTimeout() will wait until the first/'oldest' task in the TestEventScheduler has a timestamp less
   * than the current timestamp OR until the timeout is hit. If there already exists such a task, then WaitForEvent()
   * will not wait, and will immediately remove that task, run it and then return. If the timeout is hit,
   * WaitForEventWithTimeout() will return without running a task.
   *
   * @param[in] timeout The maxiumum amount to wait for an event to be run.
   * @return
   *   - true: If an event/task was succesfully run.
   *   - false: If an event/task was not run, i.e the timeout was hit.
   */
  bool WaitForEventWithTimeout(uint64_t timeout);

 private:
  EventQueue mEventQueue;
  std::string mName;
  std::atomic<EventSchedulerState> mState;
};
