/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <atomic>
#include <map>
#include <memory>
#include <regex>

using ParamMap = std::map<std::string, std::string>;

class MockResponse : public std::enable_shared_from_this<MockResponse> {
 public:
  struct MockResponseContent {
    std::vector<char> body;
    std::map<std::string, std::string> headers;
    uint32_t statusCode;

    MockResponseContent(const std::string& body = "",
      const std::map<std::string, std::string>& headers = std::map<std::string, std::string>(),
      uint32_t statusCode = 200)
        : body(body.begin(), body.end()), headers(headers), statusCode(statusCode) {}
  };
  using MockResponseHandler = std::function<MockResponseContent(const std::string& url, ttv::HttpRequestType type,
    const std::vector<ttv::HttpParam>& headerParams, const std::string& body)>;
  MockResponse(const std::string& url);

  MockResponse& SetUrl(const std::string& url);
  MockResponse& SetType(ttv::HttpRequestType type);
  MockResponse& SetRequestBody(const std::string& body, bool enableStrictBody = true);
  MockResponse& AddRequestRegex(const std::regex& regex, bool enableStrictRequestRegex = true);
  MockResponse& AddRequestParam(const std::string& name, uint32_t value);
  MockResponse& AddRequestParam(const std::string& name, const std::string& value);
  MockResponse& AddRequestHeader(const std::string& name, const std::string& value);
  MockResponse& AddJsonValue(
    const std::vector<std::string>& keyPath, const ttv::json::Value& value, bool enableStrictJsonValue = true);
  MockResponse& StrictRequestParams(bool strict);
  MockResponse& StrictRequestRegex(bool strict);
  MockResponse& StrictHeaderParams(bool strict);
  MockResponse& StrictBody(bool strict);
  MockResponse& StrictJsonValue(bool strict);

  MockResponse& SetStatusCode(uint statusCode);
  MockResponse& SetResponseHandler(const MockResponseHandler& handler);
  MockResponse& SetResponseBody(const ttv::json::Value& data);
  MockResponse& SetResponseBody(const char* data);
  MockResponse& SetResponseBody(const std::string& data);
  MockResponse& SetResponseBody(const std::vector<char>& data);
  MockResponse& SetResponseBodyFromFile(const std::string& path);
  MockResponse& AddResponseHeader(const std::string& name, const std::string& value);
  MockResponse& SetDelay(uint64_t milliseconds);

  bool IsStrictRequestParams() const { return mStrictRequestParams; }
  bool IsStrictRequestRegex() const { return mStrictRequestRegex; }
  bool IsStrictHeaderParams() const { return mStrictHeaderParams; }
  bool IsStrictBody() const { return mStrictBody; }
  bool IsStrictJsonValue() const { return mStrictJsonValues; }

  std::shared_ptr<MockResponse> Done();

  bool IsMatch(const std::string& url, ttv::HttpRequestType type, const std::vector<ttv::HttpParam>& headerParams,
    const std::string& requestBody, const ttv::json::Value& jroot) const;
  bool IsMatch(const std::string& url, ttv::HttpRequestType type, const ParamMap& headerParams,
    const std::string& requestBody, const ttv::json::Value& jroot) const;

  TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
    const std::vector<ttv::HttpParam>& headerParams, const uint8_t* requestBody, size_t requestBodySize,
    ttv::HttpRequestType httpReqType, uint timeOutInSecs, ttv::HttpRequestHeadersCallback headersCallback,
    ttv::HttpRequestCallback responseCallback, void* userData);

  int GetId() const { return mId; }

  int GetRequestCount() const { return mRequestCount; }

  uint64_t GetDelayMilliseconds() const { return mDelayMilliseconds; }
  void ResetRequestCount();
  void AssertRequestsMade();
  void AssertNoRequestsMade();

 private:
  static int smNextMockRequestId;

  int mId;
  std::atomic<int> mRequestCount;
  uint64_t mDelayMilliseconds;

  // Matching
  ttv::Uri mUrl;
  ttv::HttpRequestType mType;
  std::string mRequestBody;
  std::vector<std::regex> mRequestRegex;
  ParamMap mHeaderParams;
  std::map<std::vector<std::string>, ttv::json::Value> mJsonValues;
  bool mStrictRequestParams;
  bool mStrictRequestRegex;
  bool mStrictHeaderParams;
  bool mStrictBody;
  bool mStrictJsonValues;

  // Response
  MockResponseContent mResponseContent;
  MockResponseHandler mResponseHandler;
};

class TestHttpRequest : public ttv::HttpRequest {
 public:
  TestHttpRequest();
  virtual ~TestHttpRequest();

  MockResponse& AddResponse(const std::string& url);
  void RemoveResponse(std::shared_ptr<MockResponse> mock);

  void SetValidate(bool validate) { mValidate = validate; }
  void SetOffline(bool offline) { mOffline = offline; }

  // HttpRequest implementation
  virtual TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
    const std::vector<ttv::HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
    ttv::HttpRequestType httpReqType, uint timeOutInSecs, ttv::HttpRequestHeadersCallback headersCallback,
    ttv::HttpRequestCallback responseCallback, void* userData);

 private:
  std::map<int, std::shared_ptr<MockResponse>>::iterator FindResponse(const std::string& url, ttv::HttpRequestType type,
    const std::vector<ttv::HttpParam>& requestHeaders, const std::string& requestBody);
  std::map<int, std::shared_ptr<MockResponse>> mHttpResponses;

  std::unique_ptr<ttv::IMutex> mResponseMutex;
  bool mValidate;
  std::atomic_bool mOffline;  // If true then all requests will fail.
};
