/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"

#include "testeventschedulerfactory.h"
#include "testsocketfactory.h"
#include "testsystemclock.h"
#include "testutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/threadedeventschedulerfactory.h"

using namespace ttv;

void ttv::test::SdkBaseTest::SetUp() {
  SetUpStubs();
  SetUpComponents();
}

void ttv::test::SdkBaseTest::TearDown() {
  TearDownComponents();
  TearDownStubs();
}

void ttv::test::SdkBaseTest::SetUpStubs() {
  mMainEventSchedulerFactory = std::make_shared<ttv::TestEventSchedulerFactory>();
  ttv::SetMainEventSchedulerFactory(mMainEventSchedulerFactory);

  mBackgroundEventSchedulerFactory = std::make_shared<ttv::ThreadedEventSchedulerFactory>();
  ttv::SetBackgroundEventSchedulerFactory(mBackgroundEventSchedulerFactory);

  mHttpRequest = std::make_shared<TestHttpRequest>();

  mUserTestUtility.SetUpStubs(mHttpRequest);
  mPubSubTestUtility.SetUpStubs();

  mPreviousHttpRequest = ttv::GetHttpRequest();
  ttv::SetHttpRequest(mHttpRequest);

  mTestSocketFactory = std::make_shared<TestSocketFactory>();
  RegisterSocketFactory(mTestSocketFactory);

  mPreviousSystemClock = ttv::GetSystemClock();
  mTestSystemClock = std::make_shared<TestSystemClock>(mPreviousSystemClock);
  SetSystemClock(mTestSystemClock);
}

void ttv::test::SdkBaseTest::SetUpComponents() {}

void ttv::test::SdkBaseTest::TearDownStubs() {
  if (mTestSocketFactory != nullptr) {
    UnregisterSocketFactory(mTestSocketFactory);
    mTestSocketFactory.reset();
  }

  if (mPreviousSystemClock != nullptr) {
    ttv::SetSystemClock(mPreviousSystemClock);
    mPreviousSystemClock = nullptr;
  }

  ttv::SetHttpRequest(mPreviousHttpRequest);
  mPubSubTestUtility.TearDownStubs();

  ttv::SetMainEventSchedulerFactory(nullptr);
  mMainEventSchedulerFactory.reset();
}

void ttv::test::SdkBaseTest::TearDownComponents() {
  ShutdownTaskRunner(mTaskRunner);

  // Shutdown components first
  for (auto component : mComponents) {
    ShutdownComponent(component);
  }

  // Shutdown all modules
  std::reverse(mModules.begin(), mModules.end());
  ShutdownModules(mModules);
}

void ttv::test::SdkBaseTest::AddModule(std::shared_ptr<ttv::IModule> module) {
  mModules.push_back(module);
}

void ttv::test::SdkBaseTest::RemoveModule(std::shared_ptr<ttv::IModule> module) {
  auto iter = std::find(mModules.begin(), mModules.end(), module);
  if (iter != mModules.end()) {
    mModules.erase(iter);
  }
}

void ttv::test::SdkBaseTest::AddComponent(std::shared_ptr<ttv::Component> component) {
  mComponents.push_back(component);
}

void ttv::test::SdkBaseTest::RemoveComponent(std::shared_ptr<ttv::Component> component) {
  auto iter = std::find(mComponents.begin(), mComponents.end(), component);
  if (iter != mComponents.end()) {
    mComponents.erase(iter);
  }
}

std::shared_ptr<TaskRunner> ttv::test::SdkBaseTest::CreateTaskRunner() {
  if (mTaskRunner == nullptr) {
    mTaskRunner = std::make_shared<TaskRunner>();
  }

  return mTaskRunner;
}

std::shared_ptr<UserRepository> ttv::test::SdkBaseTest::CreateUserRepository() {
  if (mUserRepository == nullptr) {
    mUserRepository = std::make_shared<UserRepository>();
    mUserRepository->SetTaskRunner(CreateTaskRunner());

    AddComponent(mUserRepository);
  }

  return mUserRepository;
}

std::function<void()> ttv::test::SdkBaseTest::GetDefaultUpdateFunc() {
  return [this]() { Update(); };
}

void ttv::test::SdkBaseTest::Update() {
  auto scheduler = mMainEventSchedulerFactory->GetMainEventScheduler();
  if (scheduler != nullptr) {
    scheduler->WaitForEventWithTimeout(0);
  }

  UpdateTaskRunner();
  UpdateComponents();
  UpdateModules();
}

void ttv::test::SdkBaseTest::UpdateModules() {
  for (auto module : mModules) {
    module->Update();
  }
}

void ttv::test::SdkBaseTest::UpdateComponents() {
  for (auto component : mComponents) {
    component->Update();
  }
}

void ttv::test::SdkBaseTest::UpdateTaskRunner() {
  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }
}

TTV_ErrorCode ttv::test::SdkBaseTest::LogIn(
  const std::shared_ptr<CoreAPI>& coreApi, const std::string& oauth, const UserInfo& userInfo) {
  mUserTestUtility.RegisterUserInfo(oauth, userInfo);
  auto response = mHttpRequest->AddResponse("https://api.twitch.tv/kraken?oauth_token=" + oauth)
                    .SetResponseBodyFromFile("core/validate_oauth_valid.json")
                    .Done();

  bool loginComplete = false;
  TTV_ErrorCode callbackResult;
  auto ec = coreApi->LogIn(oauth,
    [&loginComplete, &callbackResult, &userInfo](const ErrorDetails& errorDetails, const UserInfo& loginUserInfo) {
      EXPECT_TRUE(TTV_SUCCEEDED(errorDetails.ec));
      EXPECT_EQ(userInfo, loginUserInfo);
      callbackResult = errorDetails.ec;
      loginComplete = true;
    });

  if (TTV_SUCCEEDED(ec)) {
    EXPECT_TRUE(WaitUntilResult(1000, [&loginComplete]() { return loginComplete; }));
    return callbackResult;
  }

  return ec;
}

TTV_ErrorCode ttv::test::SdkBaseTest::LogOut(const std::shared_ptr<CoreAPI>& coreApi, const UserInfo& userInfo) {
  bool loginComplete = false;
  TTV_ErrorCode callbackResult;
  auto ec = coreApi->LogOut(userInfo.userId, [&loginComplete, &callbackResult](TTV_ErrorCode ec) {
    EXPECT_TRUE(TTV_SUCCEEDED(ec));
    callbackResult = ec;
    loginComplete = true;
  });

  if (TTV_SUCCEEDED(ec)) {
    EXPECT_TRUE(WaitUntilResult(1000, [&loginComplete]() { return loginComplete; }));

    return callbackResult;
  }
  return ec;
}

bool ttv::test::SdkBaseTest::WaitUntilResult(uint waitUntilMilliSeconds, std::function<bool()> checkForResults) {
  std::function<void()> flushEvents = [this] { Update(); };

  return ttv::test::WaitUntilResultWithPollTask(waitUntilMilliSeconds, checkForResults, flushEvents);
}
