/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "testeventscheduler.h"

#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/tracer.h"

namespace {
const char* kLoggerName = "TestEventScheduler";
}

ttv::test::TestEventScheduler::TestEventScheduler()
    : mEventQueue("TestEventScheduler"),
      mName("TestEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::Running) {}

ttv::test::TestEventScheduler::TestEventScheduler(const std::string& name)
    : mEventQueue("TestEventScheduler-" + name),
      mName(name + "-TestEventScheduler(" + ttv::PointerToString(this) + ")"),
      mState(EventSchedulerState::Running) {}

ttv::test::TestEventScheduler::~TestEventScheduler() {
  TTV_ASSERT(mState != EventSchedulerState::Running);
  if (mState == EventSchedulerState::Running) {
    Shutdown(nullptr);
  }
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "TestEventScheduler \"%s\" destroyed", mName.c_str());
}

ttv::Result<ttv::TaskId> ttv::test::TestEventScheduler::ScheduleTask(TaskParams&& taskParams) {
  if (mState != EventSchedulerState::Running) {
    return MakeErrorResult(TTV_EC_NOT_INITIALIZED);
  }

  return MakeSuccessResult(mEventQueue.InsertTask(std::move(taskParams)));
}

bool ttv::test::TestEventScheduler::WaitForEventWithTimeout(uint64_t timeout) {
  if (mState == EventSchedulerState::Running) {
    return mEventQueue.WaitForEventWithTimeout(timeout);
  }

  return false;
}

TTV_ErrorCode ttv::test::TestEventScheduler::CancelTask(TaskId taskId) {
  if (mState != EventSchedulerState::Running) {
    return TTV_EC_NOT_INITIALIZED;
  }

  bool removed = mEventQueue.RemoveTask(taskId);

  if (!removed) {
    return TTV_EC_OPERATION_FAILED;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestEventScheduler::Shutdown(TaskFunc&& shutdownTask) {
  EventSchedulerState expected = EventSchedulerState::Running;
  if (!mState.compare_exchange_strong(expected, EventSchedulerState::ShutDown)) {
    return TTV_EC_NOT_INITIALIZED;
  }

  mEventQueue.Clear();

  if (shutdownTask != nullptr) {
    shutdownTask();
  }

  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "TestEventScheduler \"%s\" shutdown", mName.c_str());

  return TTV_EC_SUCCESS;
}

ttv::IEventScheduler::EventSchedulerState ttv::test::TestEventScheduler::GetState() {
  return mState;
}
