/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "testsocket.h"

#include "twitchsdk/core/thread.h"

#include "gtest/gtest.h"

ttv::test::TestSocket::TestSocket()
    : mTotalSent(0), mTotalReceived(0), mMaxBitrateKbps(0), mConnected(false), mShouldFail(false) {
  ttv::CreateMutex(mMutex, "TestSocket");
}

TTV_ErrorCode ttv::test::TestSocket::Connect() {
  if (mShouldFail) {
    return TTV_EC_SOCKET_CONNECT_FAILED;
  }

  if (mConnected) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mConnected = true;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestSocket::Disconnect() {
  if (!mConnected) {
    return TTV_EC_SOCKET_EALREADY;
  }

  mConnected = false;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::test::TestSocket::Send(const uint8_t* buffer, size_t length, size_t& sent) {
  sent = 0;

  if (!mConnected) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  if (mShouldFail) {
    Disconnect();
    return TTV_EC_SOCKET_RECV_ERROR;
  }

  mTotalSent += length;

  // Emulate a limited bandwidth
  if (mMaxBitrateKbps > 0) {
    uint64_t duration = 1000 * static_cast<uint64_t>(length * 8) / mMaxBitrateKbps;

    Sleep(duration);
  }

  if (mSendCallback != nullptr) {
    return mSendCallback(buffer, length, sent);
  } else {
    // TODO: We might want to validate here
    return TTV_EC_SUCCESS;
  }
}

TTV_ErrorCode ttv::test::TestSocket::Recv(uint8_t* buffer, size_t length, size_t& received) {
  received = 0;

  if (!mConnected) {
    return TTV_EC_SOCKET_ENOTCONN;
  }

  if (mShouldFail) {
    Disconnect();
    return TTV_EC_SOCKET_RECV_ERROR;
  }

  QueueEntry entry;
  TTV_ErrorCode ec = TTV_EC_SOCKET_EWOULDBLOCK;

  bool enqueued = false;

  {
    AutoMutex lock(mMutex.get());

    if (!mReceivedPayloadQueue.empty()) {
      entry = mReceivedPayloadQueue.front();
      mReceivedPayloadQueue.pop();
      enqueued = true;
    }
  }

  // There is enqueued data
  if (enqueued) {
    size_t size = std::min(entry.first.size(), length);

    if (entry.first.size() <= length) {
      memcpy(buffer, entry.first.data(), size);
      received = size;
      ec = TTV_EC_SUCCESS;
    } else {
      // TODO: We need to figure out how to handle this.  Buffer the unreceived bytes?
      EXPECT_FALSE(true);
    }
  }
  // Call a registered receive callback
  else if (mRecvCallback != nullptr) {
    ec = mRecvCallback(buffer, length, received);
  }

  mTotalReceived += static_cast<uint64_t>(received);

  if (enqueued) {
    if (entry.second != nullptr) {
      entry.second();
    }
  }

  return ec;
}

uint64_t ttv::test::TestSocket::TotalSent() {
  return mTotalSent;
}

uint64_t ttv::test::TestSocket::TotalReceived() {
  return mTotalReceived;
}

bool ttv::test::TestSocket::Connected() {
  return mConnected;
}
