/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/experiment/experimenttypes.h"

namespace ttv {
class TaskRunner;

namespace experiment {
class ExperimentAPI;
}
}  // namespace ttv

/**
 * Provides the client-side mini-experiment functionality.
 *
 * The following properties must be set before calling Initialize().
 * - SetCoreApi()
 *
 *
 * The active experiments can be set in this API instance by
 * - Fetching it via FetchExperiments(), or
 * - Loading some existing JSON using DeserializeExperiments() and then calling SetExperiments().
 *   (You may use the serialized field of ExperimentSet to save the JSON for later use)
 *
 * Once experiments have been loaded you can ask which bucket a user is in via DetermineBucket().
 *
 * If debugging your app and want to force certain experiments use SetOverride() and RemoveOverride().
 */
class ttv::experiment::ExperimentAPI : public ttv::ModuleBase {
 public:
  using FetchExperimentsCallback = std::function<void(TTV_ErrorCode ec, const std::shared_ptr<ExperimentSet>& result)>;

 public:
  ExperimentAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);

  /**
   * Set the DeviceId for analytics. This is used for ttv::experiments::Type::DeviceId type experiments.
   */
  TTV_ErrorCode SetDeviceId(const std::string& id);

  /**
   * Set the UserId for analytics. This is used for ttv::Experiments::Type::User id type experiments.
   */
  TTV_ErrorCode SetUserId(ttv::UserId userId);

  /**
   * Requests the current set of experiments from the backend.
   */
  TTV_ErrorCode FetchExperiments(const FetchExperimentsCallback& callback);
  /**
   * Sets the active experiments.
   */
  TTV_ErrorCode SetExperiments(const ExperimentSet& experiments);
  /**
   * Parses the given experiments JSON.  This is used to parse previously fetched ExperimentSet::serialized.
   */
  TTV_ErrorCode DeserializeExperiments(const std::string& json, ExperimentSet& result);
  /**
   * Determines which bucket should be used based on the device id for the given experiment
   * GUID.  The data must have previously been fetched via FetchExperiments() or SetExperiments().
   * SetDeviceId or SetUserId must be called before using this method.
   */
  TTV_ErrorCode DetermineBucket(const std::string& experiment, std::string& result);
  /**
   * Returns the version for the given experiment GUID. The data must have been previously
   * fetched via FetchExperiments() or SetExperiments().
   */
  TTV_ErrorCode GetVersion(const std::string& experimentGuid, int32_t& version);
  /**
   * Return the type for the given experiment GUID. The data must have been previously
   * fetched via FetchExperiments() or SetExperiments().
   */
  TTV_ErrorCode GetType(const std::string& experimentGuid, ExperimentType& type);
  /**
   * Return the shuffle id for the given experiment GUID. The data must have been previously
   * fetched via FetchExperiments() or SetExperiments().
   */
  TTV_ErrorCode GetShuffleId(const std::string& experimentGuid, int32_t& shuffleId);

  /**
   * Sets an explicit group to use for an experiment, typically used for debugging.
   */
  TTV_ErrorCode SetOverride(const std::string& experimentGuid, const std::string& groupValue);
  /**
   * Determines the value of the group if it is overidden.  This will be "" if not overidden.
   */
  TTV_ErrorCode GetOverride(const std::string& experimentGuid, std::string& result);
  /**
   * Retrieves all groups that have been overridden and their assigned value.
   */
  TTV_ErrorCode GetOverrides(std::map<std::string, std::string>& result);
  /**
   * Removes an explicit override set using SetOverride().
   */
  TTV_ErrorCode RemoveOverride(const std::string& experimentGuid);
  /**
   * Clears all overrides.
   */
  TTV_ErrorCode ClearOverrides();

 protected:
  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  class CoreApiClient : public ICoreApiClient {
   public:
    // ICoreApiClient implementation
    virtual std::string GetClientName();
  };

  std::shared_ptr<CoreAPI> mCoreApi;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::shared_ptr<CoreApiClient> mCoreApiClient;

  ExperimentSet mExperimentSet;  // The last fetched experiment set.

  // Unique ids used for hashing and generating buckets for experiments.
  std::string mDeviceId;
  ttv::UserId mUserId;

  std::map<std::string, std::string> mOverrides;
};
