/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <map>
#include <string>
#include <vector>

namespace ttv {
namespace experiment {
struct Group {
  std::string value;  // The name of the group used to identify the variation used.
  uint32_t weight;    // The weight assigned to this group.
};

enum class ExperimentType : int32_t {
  DeviceId = 1,   // Experiment is controlled by the unique device id. Use ExperimentAPI::SetDeviceId() to set.
  UserId = 2,     // Experiment is controlled by the unique Twitch user id. Use ExperimentAPI::SetUserId() to set.
  ChannelId = 3,  // Experiment is controlled by the currently playing channel. Not supported currently.
};

inline std::string TypeToString(ExperimentType type) {
  switch (type) {
    case ExperimentType::DeviceId:
      return "device_id";
    case ExperimentType::UserId:
      return "user_id";
    case ExperimentType::ChannelId:
      return "channel_id";
    default:
      return "";
  }
}

inline ExperimentType ToExperimentType(int32_t intType) {
  switch (intType) {
    case static_cast<int32_t>(ExperimentType::DeviceId):
      return ExperimentType::DeviceId;
    case static_cast<int32_t>(ExperimentType::UserId):
      return ExperimentType::UserId;
    case static_cast<int32_t>(ExperimentType::ChannelId):
      return ExperimentType::ChannelId;
    default:
      return ExperimentType::DeviceId;
  }
}

struct ExperimentData {
  std::string guid;           // The unique ID of the experiment.
  std::string name;           // The human-readable name of the experiment.
  int32_t version;            // The version number of the experiment.
  ExperimentType type;        // The type of the experiment.
  int32_t shuffleId;          // The Shuffle ID of the experiment
  std::vector<Group> groups;  // The available groups in the experiment.
};

struct ExperimentSet {
  std::map<std::string, ExperimentData> experiments;  // The available experiments, mapping GUID to experiment.
  std::string serialized;  // The raw experiment JSON for use in persisting the experiments across applications runs.
};
}  // namespace experiment
}  // namespace ttv
