#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/experiment/experimentapi.h"

using namespace ttv;
using namespace ttv::experiment;

namespace {
const char* kTraceComponents[] = {"ExperimentAPI"};

std::shared_ptr<ExperimentAPI> gExperimentAPI;
std::shared_ptr<Context> gContext;  // Cached by the static initializer

void Print(const ExperimentSet& set, std::string indent) {
  std::cout << indent << "ExperimentSet:" << std::endl;
  indent += "  ";

  for (const auto& kvp : set.experiments) {
    const auto& guid = kvp.first;
    const auto& experiment = kvp.second;

    std::cout << indent << "Experiment: " << experiment.name << " (" << guid << ")" << std::endl;
    indent += "  ";

    for (const auto& group : experiment.groups) {
      std::cout << indent << group.value << ", " << group.weight << std::endl;
    }
  }
}

std::shared_ptr<CommandCategory> RegisterTrackingCommands(std::shared_ptr<Context> context) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Experiment", "The ttv::experiment::ExperimentAPI command interface");

  category->AddCommand("ExperimentInit")
    .AddFunction()
    .Description("Calls ExperimentAPI::Initialize")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->Initialize([](TTV_ErrorCode ec) {
        std::cout << "ExperimentInit completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ExperimentShutdown")
    .AddFunction()
    .Description("Calls ExperimentAPI::Shutdown")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "ExperimentShutdown completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("ExperimentFetch")
    .AddFunction()
    .Description("Calls ExperimentAPI::FetchExperiments")
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gExperimentAPI->FetchExperiments([](TTV_ErrorCode ec, const std::shared_ptr<ExperimentSet>& result) {
          if (TTV_SUCCEEDED(ec)) {
            Print(*result, "");
          } else {
            std::cout << "ExperimentAPI::FetchExperiments async failure" << std::endl;
          }
        });
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ExperimentDetermineBucket")
    .AddFunction()
    .Description("Calls ExperimentAPI::DetermineBucket")
    .AddParam("experiment", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string result;

      TTV_ErrorCode ec = gExperimentAPI->DetermineBucket(params[0], result);
      ReportCommandResult(ec);

      if (TTV_SUCCEEDED(ec)) {
        std::cout << params[0].GetString() << " --> " << result << std::endl;
      }

      return true;
    });

  category->AddCommand("ExperimentSetOverride")
    .AddFunction()
    .Description("Calls ExperimentAPI::SetOverride")
    .AddParam("experiment", ParamType::String)
    .AddParam("value", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->SetOverride(params[0], params[1]);
      ReportCommandResult(ec);

      return true;
    });

  category->AddCommand("ExperimentGetOverride")
    .AddFunction()
    .Description("Calls ExperimentAPI::GetOverride")
    .AddParam("experiment", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::string result;

      TTV_ErrorCode ec = gExperimentAPI->GetOverride(params[0], result);
      ReportCommandResult(ec);

      if (TTV_SUCCEEDED(ec)) {
        std::cout << params[0].GetString() << " --> " << result << std::endl;
      }

      return true;
    });

  category->AddCommand("ExperimentGetOverrides")
    .AddFunction()
    .Description("Calls ExperimentAPI::GetOverrides")
    .AddParam("experiment", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::map<std::string, std::string> result;

      TTV_ErrorCode ec = gExperimentAPI->GetOverrides(result);
      ReportCommandResult(ec);

      if (TTV_SUCCEEDED(ec)) {
        for (const auto& kvp : result) {
          std::cout << kvp.first << " --> " << kvp.second << std::endl;
        }
      }

      return true;
    });

  category->AddCommand("ExperimentRemoveOverride")
    .AddFunction()
    .Description("Calls ExperimentAPI::RemoveOverride")
    .AddParam("experiment", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->RemoveOverride(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SetDeviceId")
    .AddFunction()
    .Description("Calls ExperimentAPI::SetDeviceId")
    .AddParam("deviceId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->SetDeviceId(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SetUserId")
    .AddFunction()
    .Description("Calls ExperimentAPI::SetUserId")
    .AddParam("userId", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gExperimentAPI->SetUserId(params[0]);
      ReportCommandResult(ec);
      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      gContext = context;

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));
      gExperimentAPI = std::make_shared<ExperimentAPI>();
      gExperimentAPI->SetCoreApi(coreApi);
      context->RegisterModule(gExperimentAPI);

      auto commands = RegisterTrackingCommands(context);
      context->RegisterCommandCategory(commands);
    },
    0);
});
}  // namespace
