/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/experiment/internal/pch.h"

#include "twitchsdk/experiment/experimentapi.h"

#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/experiment/experimentapiutils.h"
#include "twitchsdk/experiment/internal/task/fetchexperimentstask.h"

namespace {
using namespace ttv;
using namespace ttv::experiment;

const char* kModuleName = "ttv::experiment::ExperimentAPI";
}  // namespace

ttv::experiment::ExperimentAPI::ExperimentAPI() : mUserId(0) {}

std::string ttv::experiment::ExperimentAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState == IModule::State::Uninitialized) {
    mCoreApi = coreApi;
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_ALREADY_INITIALIZED;
  }
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::SetDeviceId(const std::string& id) {
  if (id.empty()) {
    return TTV_EC_INVALID_ARG;
  }

  mDeviceId = id;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::SetUserId(ttv::UserId id) {
  mUserId = id;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::Initialize(const InitializeCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);
  TTV_RETURN_ON_NULL(mCoreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;

    mCoreApiClient = std::make_shared<CoreApiClient>();
    mMainEventScheduler = mCoreApi->GetMainEventScheduler();
    mTaskRunner = std::make_shared<TaskRunner>();
  }

  ec = mCoreApi->RegisterClient(mCoreApiClient);
  if (TTV_SUCCEEDED(ec)) {
    NotifyStateChange();
    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);

  if (TTV_SUCCEEDED(ec)) {
    RegisterShutdownCallback(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    case State::Initialized: {
      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::experiment::ExperimentAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::experiment::ExperimentAPI::CompleteShutdown() {
  if (mCoreApiClient != nullptr) {
    if (mCoreApi != nullptr) {
      mCoreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  mCoreApi.reset();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::FetchExperiments(const FetchExperimentsCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  auto task = std::make_shared<FetchExperimentsTask>(
    [this, callback](FetchExperimentsTask* /*source*/, TTV_ErrorCode ec, const std::shared_ptr<ExperimentSet>& result) {
      // Keep a deep copy around
      if (TTV_SUCCEEDED(ec)) {
        mExperimentSet = *result;
      }

      if (callback != nullptr) {
        callback(ec, result);
      }
    });

  bool submitted = mTaskRunner->AddTask(task);
  if (submitted) {
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_SHUTTING_DOWN;
  }
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::SetExperiments(const ExperimentSet& experiments) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mExperimentSet = experiments;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::DeserializeExperiments(const std::string& json, ExperimentSet& result) {
  return FetchExperimentsTask::ParseExperimentSet(json, result);
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::GetVersion(const std::string& experimentGuid, int32_t& version) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  if (mDeviceId.empty() && mUserId == 0) {
    return TTV_EC_INVALID_STATE;
  }

  // Make sure it's a valid experiment
  auto iter = mExperimentSet.experiments.find(experimentGuid);
  if (iter == mExperimentSet.experiments.end()) {
    return TTV_EC_INVALID_ARG;
  }

  const ExperimentData& experiment = iter->second;
  version = experiment.version;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::GetType(const std::string& experimentGuid, ExperimentType& type) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  if (mDeviceId.empty() && mUserId == 0) {
    return TTV_EC_INVALID_STATE;
  }

  // Make sure it's a valid experiment
  auto iter = mExperimentSet.experiments.find(experimentGuid);
  if (iter == mExperimentSet.experiments.end()) {
    return TTV_EC_INVALID_ARG;
  }

  const ExperimentData& experiment = iter->second;
  type = experiment.type;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::GetShuffleId(const std::string& experimentGuid, int32_t& shuffleId) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  if (mDeviceId.empty() && mUserId == 0) {
    return TTV_EC_INVALID_STATE;
  }

  // Make sure it's a valid experiment
  auto iter = mExperimentSet.experiments.find(experimentGuid);
  if (iter == mExperimentSet.experiments.end()) {
    return TTV_EC_INVALID_ARG;
  }

  const ExperimentData& experiment = iter->second;
  shuffleId = experiment.shuffleId;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::SetOverride(
  const std::string& experimentGuid, const std::string& groupValue) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_EMPTY_STDSTRING(experimentGuid, TTV_EC_INVALID_ARG);

  if (!groupValue.empty()) {
    mOverrides[experimentGuid] = groupValue;
    return TTV_EC_SUCCESS;
  } else {
    return RemoveOverride(experimentGuid);
  }
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::DetermineBucket(const std::string& experimentGuid, std::string& result) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  return ttv::experiment::utils::DetermineBucket(
    experimentGuid, mExperimentSet, mDeviceId, mUserId, mOverrides, result);
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::GetOverride(const std::string& experimentGuid, std::string& result) {
  result.clear();

  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_EMPTY_STDSTRING(experimentGuid, TTV_EC_INVALID_ARG);

  auto iter = mOverrides.find(experimentGuid);
  if (iter != mOverrides.end()) {
    result = iter->second;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::GetOverrides(std::map<std::string, std::string>& result) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  result = mOverrides;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::RemoveOverride(const std::string& experimentGuid) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_EMPTY_STDSTRING(experimentGuid, TTV_EC_INVALID_ARG);

  auto iter = mOverrides.find(experimentGuid);
  if (iter != mOverrides.end()) {
    mOverrides.erase(iter);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::ExperimentAPI::ClearOverrides() {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mOverrides.clear();

  return TTV_EC_SUCCESS;
}

std::string ttv::experiment::ExperimentAPI::CoreApiClient::GetClientName() {
  return kModuleName;
}
