/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/experiment/internal/pch.h"

#include "twitchsdk/experiment/experimentapiutils.h"

#include "twitchsdk/core/sha1.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/experiment/experimenttypes.h"

using namespace ttv;
using namespace ttv::experiment;

TTV_ErrorCode ttv::experiment::utils::DetermineBucket(const std::string& experimentGuid,
  const ExperimentSet& experimentSet, const std::string& deviceId, const ttv::UserId& userId,
  const std::map<std::string, std::string>& overrides, std::string& result) {
  result.clear();

  // Check for an explicit value set by the app for debugging
  auto overrideIter = overrides.find(experimentGuid);
  if (overrideIter != overrides.end()) {
    result = overrideIter->second;
    return TTV_EC_SUCCESS;
  }

  // Make sure it's a valid experiment
  auto iter = experimentSet.experiments.find(experimentGuid);
  if (iter == experimentSet.experiments.end()) {
    return TTV_EC_INVALID_ARG;
  }

  const ExperimentData& experiment = iter->second;
  if (experiment.groups.empty()) {
    return TTV_EC_NOT_AVAILABLE;
  }

  // Determine the total weight
  double totalWeight = 0.0;
  bool anyPositiveWeight = false;
  for (const auto& group : experiment.groups) {
    if (group.weight > 0.0) {
      totalWeight += static_cast<double>(group.weight);
      anyPositiveWeight = true;
    }
  }

  if (!anyPositiveWeight) {
    result = experiment.groups.front().value;
    return TTV_EC_SUCCESS;
  }

  // Figure out the proper analyticsId if using device or user
  std::string analyticsId;
  if (experiment.type == ExperimentType::DeviceId) {
    if (deviceId.empty()) {
      return TTV_EC_INVALID_STATE;
    }

    analyticsId = deviceId;
  } else if (experiment.type == ExperimentType::UserId) {
    analyticsId = std::to_string(userId);
  }

  // Hash the unique id and experiment
  SHA1 sha1;
  std::string key = experiment.guid + analyticsId + std::to_string(experiment.shuffleId);
  sha1.Update(key);
  std::string hash = sha1.Final();

  uint32_t unsignedHash = SHA1::HashAsUInt32(hash);

  // NOTE: The groups are sorted by "value" so ordering is consistent between fetches
  double currWeight = static_cast<double>(unsignedHash) / static_cast<double>(std::numeric_limits<uint32_t>::max());
  for (const auto& group : experiment.groups) {
    currWeight -= static_cast<double>(group.weight) / totalWeight;

    if (currWeight <= 0) {
      result = group.value;
      return TTV_EC_SUCCESS;
    }
  }

  // Failsafe, choose the last bucket
  result = experiment.groups.back().value;
  return TTV_EC_SUCCESS;
}
