/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/experiment/internal/pch.h"

#include "twitchsdk/experiment/internal/task/fetchexperimentstask.h"

#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/experiment/experimenttypes.h"

// curl "https://www.twitch.tv/experiments.json" | python -m json.tool

namespace {
using namespace ttv;
using namespace ttv::experiment;

const char* kExperimentsURL = "https://www.twitch.tv/experiments.json";
const char* kExperimentsNameKey = "name";      // "experiment_name"
const char* kExperimentsVersionKey = "v";      // "experiment_version"
const char* kExperimentsShuffleIDKey = "s";    // Shuffle ID - Shuffle user treatment assignment
const char* kExperimentsTypeKey = "t";         // "experiment_type"
const char* kExperimentsGroupsKey = "groups";  // "experiment_group"
const char* kExperimentsWeightKey = "weight";
const char* kExperimentsValueKey = "value";
}  // namespace

ttv::experiment::FetchExperimentsTask::FetchExperimentsTask(Callback callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(callback) {
  ttv::trace::Message("TrackingRequestTask", MessageLevel::Info, "TrackingRequestTask created");
}

void ttv::experiment::FetchExperimentsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kExperimentsURL;
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::experiment::FetchExperimentsTask::ProcessResponse(uint statusCode, const std::vector<char>& response) {
  if (!Is2XX(statusCode)) {
    mTaskStatus = TTV_EC_API_REQUEST_FAILED;
  } else if (response.size() > 0) {
    mResult = std::make_shared<ExperimentSet>();
    std::string str(response.begin(), response.end());
    mTaskStatus = ParseExperimentSet(str, *mResult);
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

TTV_ErrorCode ttv::experiment::FetchExperimentsTask::ParseExperimentSet(
  const std::string& json, ExperimentSet& result) {
  // Parse the returned JSON
  json::Value jsonRoot;
  json::Reader jsonReader;
  bool parsed = jsonReader.parse(json, jsonRoot);
  if (!parsed) {
    ttv::trace::Message("FetchExperimentsTask", MessageLevel::Error,
      "FetchExperimentsTask::ParseExperimentSet - JSON parsing failed. json string is %s", json.c_str());
    return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }

  result.serialized = json;

  for (auto iter = jsonRoot.begin(); iter != jsonRoot.end(); ++iter) {
    ExperimentData experiment;
    experiment.guid = iter.key().asString();

    TTV_ErrorCode ec = ParseExperiment(*iter, experiment);
    if (TTV_FAILED(ec)) {
      ttv::trace::Message("FetchExperimentsTask", MessageLevel::Warning,
        "FetchExperimentsTask::ParseExperimentSet - Failed to parse experiment: %s", experiment.guid.c_str());
      continue;
    }

    result.experiments[experiment.guid] = std::move(experiment);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::experiment::FetchExperimentsTask::ParseExperiment(
  const json::Value& jsonExperiment, ExperimentData& experiment) {
  // "24d2ca90-8a02-4aae-aeff-377eca92eece":
  // {"name":"tvapps_tvos_player_native","v":7915,"t":1,"s":3,"groups":
  // [{"weight":50,"value":"control"},{"weight":50,"value":"active"}]}

  if (!jsonExperiment.isObject()) {
    return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }

  const auto& jsonName = jsonExperiment[kExperimentsNameKey];
  if (!jsonName.isString()) {
    return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
  experiment.name = jsonName.asString();

  const auto& jsonVersion = jsonExperiment[kExperimentsVersionKey];
  if (jsonVersion.isIntegral()) {
    experiment.version = static_cast<int32_t>(jsonVersion.asInt());
  } else {
    experiment.version = 0;  // Fail-safe is version is not available
  }

  const auto& jsonType = jsonExperiment[kExperimentsTypeKey];
  if (jsonType.isIntegral()) {
    experiment.type = ToExperimentType(static_cast<int32_t>(jsonType.asInt()));
  } else {
    experiment.type = ExperimentType::DeviceId;  // Legacy experiments use DeviceId
  }

  const auto& jsonShuffleId = jsonExperiment[kExperimentsShuffleIDKey];
  if (jsonShuffleId.isIntegral()) {
    experiment.shuffleId = static_cast<int32_t>(jsonShuffleId.asInt());
  } else {
    experiment.shuffleId = 0;  // ShuffleID should always be present, otherwise use default.
  }

  const auto& jsonGroupArray = jsonExperiment[kExperimentsGroupsKey];
  if (jsonGroupArray.isNull() || !jsonGroupArray.isArray()) {
    return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }

  for (const auto& jsonGroup : jsonGroupArray) {
    if (jsonGroup.isNull() || !jsonGroup.isObject()) {
      return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    const auto& jsonValue = jsonGroup[kExperimentsValueKey];
    const auto& jsonWeight = jsonGroup[kExperimentsWeightKey];

    if (!jsonValue.isString() || !jsonWeight.isNumeric()) {
      return TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    Group group;
    group.value = jsonValue.asString();
    group.weight = static_cast<uint32_t>(jsonWeight.asUInt());
    experiment.groups.emplace_back(std::move(group));
  }

  // Sort the groups by value to conserve ordering in case the response changes the order for some reason
  std::sort(experiment.groups.begin(), experiment.groups.end(),
    [](const Group& a, const Group& b) { return a.value < b.value; });

  return TTV_EC_SUCCESS;
}

void ttv::experiment::FetchExperimentsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
