/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/social/internal/task/socialfriendrequeststask.h"
#include "twitchsdk/social/internal/task/socialupdatefriendtask.h"
#include "twitchsdk/social/socialtypes.h"

#include <unordered_set>

#include <map>
#include <memory>
#include <vector>

namespace ttv {
class TaskRunner;
class Task;
class User;
class PagedRequestFetcher;

namespace social {
class FriendList;
class FriendListListenerProxy;
class SocialGetFriendsTask;
class SocialGetFriendsPresenceTask;
class SocialGetUserInfoTask;
}  // namespace social
}  // namespace ttv

// https://twitch.quip.com/VRANAAv7nosa

/**
 * Manages a user's friend list.
 */
class ttv::social::FriendList : public UserComponent {
 public:
  using UpdateFriendshipCallback =
    std::function<void(TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status)>;
  using FetchFriendListCallback = std::function<void(TTV_ErrorCode ec, const std::vector<Friend>& friends)>;
  using FetchFriendStatusCallback = std::function<void(TTV_ErrorCode ec, FriendStatus status)>;
  using FetchFriendRequestsCallback = std::function<void(TTV_ErrorCode ec, const std::vector<FriendRequest>& requests)>;
  using FetchUnreadFriendRequestCountCallback = std::function<void(TTV_ErrorCode ec, uint32_t count)>;
  using MarkAllFriendRequestsReadCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchRecommendedFriendsCallback =
    std::function<void(TTV_ErrorCode ec, const std::vector<UserInfo>& recommendedFriends)>;
  using DismissRecommendedFriendCallback = std::function<void(TTV_ErrorCode ec)>;

  class IListener {
   public:
    virtual ~IListener() = default;
    virtual void OnFriendInfoChanged(FriendList* source, const std::vector<Friend>& friends) = 0;
    virtual void OnFriendshipChanged(
      FriendList* source, const std::vector<Friend>& added, const std::vector<Friend>& removed) = 0;
    virtual void OnRealtimeFriendRequestReceived(FriendList* source, const FriendRequest& requestList) = 0;
    virtual void OnUnreadFriendRequestCountChanged(FriendList* source, uint32_t count) = 0;
    virtual void OnFriendRequestRemoved(FriendList* source, UserId otherUserId, FriendRequestRemovalReason reason) = 0;
  };

 public:
  FriendList(const std::shared_ptr<User>& user, const FeatureFlags& enabledFeatures);
  virtual ~FriendList() override;

  void SetUserRepository(std::shared_ptr<UserRepository> repository) { mUserRepository = repository; }

  std::shared_ptr<User> GetUser() { return mUser.lock(); }

  void AddListener(std::shared_ptr<IListener> listener);
  void RemoveListener(std::shared_ptr<IListener> listener);

  virtual TTV_ErrorCode Initialize() override;
  TTV_ErrorCode FetchFriendList(const FetchFriendListCallback& callback);

  TTV_ErrorCode UpdateFriendship(UserId otherUserId, FriendAction action, const UpdateFriendshipCallback& callback);
  TTV_ErrorCode FetchFriendRequests(const FetchFriendRequestsCallback& callback);

  TTV_ErrorCode FetchUnreadFriendRequestCount(const FetchUnreadFriendRequestCountCallback& callback);
  TTV_ErrorCode MarkAllFriendRequestsRead(const MarkAllFriendRequestsReadCallback& callback);

  TTV_ErrorCode FetchFriendStatus(UserId otherUserId, const FetchFriendStatusCallback& callback);

  TTV_ErrorCode FetchRecommendedFriends(const FetchRecommendedFriendsCallback& callback);
  TTV_ErrorCode DismissRecommendedFriend(UserId dismissUserId, const DismissRecommendedFriendCallback& callback);

  // Component overrides
  virtual void SetTaskRunner(std::shared_ptr<TaskRunner> taskRunner) override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  // UserComponent overrides
  virtual std::string GetLoggerName() const override;

  static std::string GetComponentName() { return "ttv::social::FriendList"; }

 private:
  class PubSubTopicListener : public PubSubClient::ITopicListener {
   public:
    PubSubTopicListener(FriendList* owner);

    // PubSubClient::ITopicListener implementation
    virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
      PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClient* source, const std::string& topic, const json::Value& msg) override;
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

   private:
    FriendList* mOwner;
  };

  struct FriendEntry {
    Friend friendData;
    uint64_t presenceUpdateIndex;
  };

  using CompleteCallback = std::function<void(TTV_ErrorCode ec)>;
  using UserIdsLookupCallback = std::function<void(UserId localUserId, UserId otherUserId, TTV_ErrorCode ec)>;

  using FriendRequestsTaskSetupCallback = std::function<void(std::shared_ptr<SocialFriendRequestsTask> task)>;
  using FriendRequestsTaskProcessResultCallback =
    std::function<TTV_ErrorCode(std::shared_ptr<SocialFriendRequestsTask::Result> result)>;

  using GetUserInfoCompleteCallback = std::function<void(TTV_ErrorCode ec)>;
  using GetUserInfoTargetCallback = std::function<UserInfo*(UserId userId)>;
  using FriendChangeSetupCallback = std::function<void(UserId ownerUserId, UserId otherUserId, TTV_ErrorCode ec)>;

  using FriendRequestMap = std::map<UserId, FriendRequest>;

  // PubSubClient::ITopicListener implementation
  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec);
  void OnTopicMessageReceived(const std::string& topic, const json::Value& msg);

  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  // UserComponent overrides
  virtual void OnUserInfoFetchComplete(TTV_ErrorCode ec) override;

  TTV_ErrorCode SubscribeTopics();
  void SetFriendPresence(UserId userId, Timestamp updateTime, uint64_t index, PresenceUserAvailability availability,
    std::unique_ptr<PresenceActivity>&& activity);

  void RequestFriendsList();
  void HandleFriendsList(std::vector<FriendEntry>&& list);

  void RequestFriendRequests();
  void RequestFriendRequestUnreadCount();

  void RequestRecommendedFriends();

  void NotifyFriendInfoChanges();
  void NotifyFriendAddsRemoves();

  TTV_ErrorCode PerformFriendRequestListManagement(FriendRequestsTaskSetupCallback setupCallback,
    FriendRequestsTaskProcessResultCallback processCallback, CompleteCallback completeCallback);
  void AddToPendingFriendAdds(UserId userId);
  void AddToPendingFriendRemoves(const FriendEntry& entry);
  bool RemoveFriendRequest(UserId userId, FriendRequestRemovalReason reason);
  TTV_ErrorCode HandleRealtimeFriendRequestReceived(UserId userId);
  TTV_ErrorCode HandleFriendRequestAccepted(UserId userId, FriendRequestRemovalReason reason);
  TTV_ErrorCode HandleFriendRequestRemoved(UserId userId, FriendRequestRemovalReason reason);
  void HandleFriendRemoved(UserId userId);

  void SetNumUnreadFriendRequests(uint32_t count);
  std::vector<Friend> BuildSocialFriendListResult(const std::unordered_set<UserId>& ids);
  std::vector<Friend> BuildSocialFriendListResult(const std::map<UserId, Friend>& users);

  std::shared_ptr<UserRepository> mUserRepository;

  std::shared_ptr<PubSubClient> mPubSub;
  std::shared_ptr<PubSubTopicListener> mPubSubTopicListener;
  std::shared_ptr<PubSubTopicListenerHelper> mPubSubTopicListenerHelper;

  std::unordered_set<UserId> mPendingAddedFriends;  //!< The set of friends who were added via a friend request.
  std::map<UserId, Friend> mPendingRemovedFriends;  //!< The set of friends who were removed from the friend list.

  std::unordered_set<UserId> mPendingInfoChanges;  //!< The list of users who have had their info changed.

  std::map<UserId, FriendEntry> mFriendMap;
  std::vector<UserInfo> mRecommendedFriends;
  CallbackQueue<FetchFriendListCallback> mFetchFriendListCallbacks;
  CallbackQueue<FetchFriendRequestsCallback> mFetchFriendRequestsCallbacks;
  CallbackQueue<FetchUnreadFriendRequestCountCallback> mFetchUnreadFriendRequestCountCallbacks;
  CallbackQueue<FetchRecommendedFriendsCallback> mFetchRecommendedFriendsCallbacks;

  FriendRequestMap mFriendRequests;
  uint32_t mNumUnreadFriendRequests;  //!< The number of unread friend requests

  WaitForExpiry mFriendListFetchTimer;
  WaitForExpiry mFriendRequestFetchTimer;
  WaitForExpiry mFriendRequestUnreadCountFetchTimer;
  WaitForExpiry mRecommendedFriendsFetchTimer;
  EventSource<IListener> mListeners;
  std::string mPresenceTopic;
  std::string mFriendshipTopic;

  FeatureFlags mEnabledFeatures;
  bool mReceivedFirstFriendListFetch;
  bool mReceivedFirstFriendRequestsFetch;
  bool mReceivedFirstFriendRequestUnreadCountFetch;
  bool mReceivedFirstRecommendedFriendsFetch;
  std::shared_ptr<PagedRequestFetcher> mFriendRequestFetcher;
};

/**
 * A lambda proxy for FriendList::IListener.
 */
class ttv::social::FriendListListenerProxy : public FriendList::IListener {
 public:
  using OnFriendInfoChangedFunc = std::function<void(FriendList* source, const std::vector<Friend>& list)>;
  using OnFriendshipChangedFunc =
    std::function<void(FriendList* source, const std::vector<Friend>& added, const std::vector<Friend>& removed)>;
  using OnRealtimeFriendRequestReceivedFunc = std::function<void(FriendList* source, const FriendRequest& request)>;
  using OnUnreadFriendRequestCountChangedFunc = std::function<void(FriendList* source, uint32_t count)>;
  using OnFriendRequestRemovedFunc =
    std::function<void(FriendList* source, UserId otherUserId, FriendRequestRemovalReason reason)>;

 public:
  // FriendList::Listener implementation
  virtual void OnFriendInfoChanged(FriendList* source, const std::vector<Friend>& list) override {
    if (mOnFriendInfoChangedFunc != nullptr) {
      mOnFriendInfoChangedFunc(source, list);
    }
  }

  virtual void OnFriendshipChanged(
    FriendList* source, const std::vector<Friend>& added, const std::vector<Friend>& removed) override {
    if (mOnFriendshipChangedFunc != nullptr) {
      mOnFriendshipChangedFunc(source, added, removed);
    }
  }

  virtual void OnRealtimeFriendRequestReceived(FriendList* source, const FriendRequest& request) override {
    if (mOnRealtimeFriendRequestReceivedFunc != nullptr) {
      mOnRealtimeFriendRequestReceivedFunc(source, request);
    }
  }

  virtual void OnUnreadFriendRequestCountChanged(FriendList* source, uint32_t count) override {
    if (mOnUnreadFriendRequestCountChangedFunc != nullptr) {
      mOnUnreadFriendRequestCountChangedFunc(source, count);
    }
  }

  virtual void OnFriendRequestRemoved(
    FriendList* source, UserId otherUserId, FriendRequestRemovalReason reason) override {
    if (mOnFriendRequestRemovedFunc != nullptr) {
      mOnFriendRequestRemovedFunc(source, otherUserId, reason);
    }
  }

 public:
  OnFriendInfoChangedFunc mOnFriendInfoChangedFunc;
  OnFriendshipChangedFunc mOnFriendshipChangedFunc;
  OnRealtimeFriendRequestReceivedFunc mOnRealtimeFriendRequestReceivedFunc;
  OnUnreadFriendRequestCountChangedFunc mOnUnreadFriendRequestCountChangedFunc;
  OnFriendRequestRemovedFunc mOnFriendRequestRemovedFunc;
};
