
/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 *********************************************************************************************/

#pragma once

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/social/socialtypes.h"

#include <map>
#include <memory>
#include <vector>

namespace ttv {
class TaskRunner;
class Task;
class PubSubClient;

namespace social {
class Presence;
class PresenceListenerProxy;
}  // namespace social
}  // namespace ttv

/**
 * Manages a local user's presence state.
 */
class ttv::social::Presence : public UserComponent {
 public:
  using PostPresenceCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchPresenceSettingsCallback = std::function<void(TTV_ErrorCode ec, const PresenceSettings& settings)>;
  using SetPresenceSettingsCallback = std::function<void(TTV_ErrorCode ec)>;

  class IListener {
   public:
    virtual ~IListener() = default;
    virtual void OnSettingsChanged(Presence* source, const PresenceSettings& settings) = 0;
  };

 public:
  Presence(std::shared_ptr<User> user);
  virtual ~Presence() override;

  void SetUserRepository(std::shared_ptr<UserRepository> repository) { mUserRepository = repository; }

  std::shared_ptr<User> GetUser() { return mUser.lock(); }

  void AddListener(std::shared_ptr<IListener> listener);
  void RemoveListener(std::shared_ptr<IListener> listener);

  virtual TTV_ErrorCode Initialize() override;
  TTV_ErrorCode PostPresence(const PostPresenceCallback& callback);

  TTV_ErrorCode SetSessionAvailability(PresenceSessionAvailability availability);

  TTV_ErrorCode AddWatchingActivity(ChannelId channelId, PresenceActivityToken& activityToken);
  TTV_ErrorCode AddPlayingActivity(
    GameId gameId, const std::string& gameDisplayContext, PresenceActivityToken& activityToken);
  TTV_ErrorCode RemoveActivity(PresenceActivityToken activityToken);

  TTV_ErrorCode SetPresenceSettings(const PresenceSettings& settings, const SetPresenceSettingsCallback& callback);
  TTV_ErrorCode FetchSettings(const FetchPresenceSettingsCallback& callback);

  TTV_ErrorCode SetAutomaticPresencePostingEnabled(bool enabled);
  TTV_ErrorCode GetAutomaticPresencePostingEnabled(bool& enabled);

  static std::string GetComponentName() { return "ttv::social::Presence"; }

  // Component overrides
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

  // UserComponent overrides
  virtual std::string GetLoggerName() const override;

 protected:
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  class PubSubTopicListener : public PubSubClient::ITopicListener {
   public:
    PubSubTopicListener(Presence* owner);

    // PubSubClient::ITopicListener implementation
    virtual void OnTopicSubscribeStateChanged(PubSubClient* source, const std::string& topic,
      PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) override;
    virtual void OnTopicMessageReceived(
      PubSubClient* source, const std::string& topic, const json::Value& msg) override;
    virtual void OnTopicListenerRemoved(PubSubClient* source, const std::string& topic, TTV_ErrorCode ec) override;

   private:
    Presence* mOwner;
  };

  // PubSubClient::ITopicListener implementation
  void OnTopicSubscribeStateChanged(
    const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec);
  void OnTopicMessageReceived(const std::string& topic, const json::Value& msg);
  void OnTopicListenerRemoved(const std::string& topic, TTV_ErrorCode ec);

  // UserComponent overrides
  virtual void OnUserInfoFetchComplete(TTV_ErrorCode ec) override;

  TTV_ErrorCode SubscribeTopics();
  TTV_ErrorCode UpdateSettings();
  void SetSettings(const PresenceSettings& settings);
  void PresenceChanged();

  uint32_t InsertActivity(std::unique_ptr<PresenceActivity>&& activity);

  std::shared_ptr<UserRepository> mUserRepository;

  std::shared_ptr<PubSubClient> mPubSub;
  std::shared_ptr<PubSubTopicListener> mPubSubTopicListener;
  std::shared_ptr<PubSubTopicListenerHelper> mPubSubTopicListenerHelper;

  std::shared_ptr<Task> mSocialSettingsTask;
  std::shared_ptr<Task> mSocialPostPresenceTask;
  EventSource<IListener> mListeners;
  WaitForExpiry mPresencePostTimer;
  WaitForExpiry mSettingsFetchTimer;
  uint64_t mPostIntervalMilliseconds;
  std::map<PresenceActivityToken, std::unique_ptr<PresenceActivity>> mActivities;
  std::string mSessionGuid;
  std::string mPresenceTopic;
  PresenceActivityToken mActivityTokenCounter;
  std::unique_ptr<PresenceSettings> mSettings;
  CallbackQueue<FetchPresenceSettingsCallback> mFetchSettingsCallbacks;
  PresenceSessionAvailability mAvailability;
  bool mEnableAutomaticPosting;
};

/**
 * A lambda proxy for Presence::IListener.
 */
class ttv::social::PresenceListenerProxy : public Presence::IListener {
 public:
  using OnSettingsChangedFunc = std::function<void(Presence* source, const PresenceSettings& settings)>;

 public:
  virtual void OnSettingsChanged(Presence* source, const PresenceSettings& settings) override {
    if (mOnSettingsChangedFunc != nullptr) {
      mOnSettingsChangedFunc(source, settings);
    }
  }

 public:
  OnSettingsChangedFunc mOnSettingsChangedFunc;
};
