/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/task/httptask.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/social/internal/task/socialjson.h"

namespace ttv {
namespace social {
class SocialFriendRequestsTask;
}
}  // namespace ttv

class ttv::social::SocialFriendRequestsTask : public ttv::HttpTask {
 public:
  static const uint32_t kMaxLimit = 500;

  enum class Action {
    Unset,
    Fetch,
    MarkAllRead,
    GetUnreadCount,
  };

  enum class SortDirection { Descending, Ascending };

  struct FriendRequestEntry {
    UserInfo userInfo;
    Timestamp requestedTimestamp;  //!< The time that the request was sent.
  };

  struct Result {
    Result();

    std::vector<FriendRequestEntry> requests;
    std::string cursor;  //!< The cursor string used to fetch the next page of results, if there are more.
    int32_t
      total;  //!< The total number of outstanding requests.  Use this to determine if there are more pages to request.
    Action action;
    SortDirection direction;
  };

  using Callback =
    std::function<void(SocialFriendRequestsTask* source, TTV_ErrorCode ec, std::shared_ptr<Result> result)>;

 public:
  SocialFriendRequestsTask(UserId userId, const std::string& authToken, Callback callback);

  void FetchRequests(uint32_t limit, SortDirection sortDirection, const std::string& cursor);
  void MarkAllRead();
  void GetUnreadCount();

 protected:
  virtual const char* GetTaskName() const override { return "SocialFriendRequestsTask"; }

  virtual void FillHttpRequestInfo(HttpRequestInfo& requestInfo) override;
  virtual bool ProcessHeaders(uint statusCode, const std::map<std::string, std::string>& headers) override;
  virtual void ProcessResponse(uint statusCode, const std::vector<char>& response) override;
  virtual void OnComplete() override;

 private:
  bool ParseFriendRequestListJson(const json::Value& jArray);

  std::shared_ptr<Result> mResult;
  Callback mCallback;
  std::string mCursor;
  Action mAction;
  SortDirection mSortDirection;
  UserId mUserId;
  uint32_t mLimit;
};
