/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/social/socialerrortypes.h"
#include "twitchsdk/social/socialtypes.h"

#include <memory>

namespace ttv {
class TaskRunner;
class UserRepository;

namespace social {
class SocialAPI;
class ISocialAPIListener;
class FriendList;
class Presence;

struct SocialAPIInternalData;
}  // namespace social
}  // namespace ttv

/**
 * Provides the main social functionality.
 *
 * The following properties must be set before calling Initialize().
 * - SetListener()
 * - SetFeatureFlags()
 */
class ttv::social::SocialAPI : public ModuleBase {
 public:
  using PostPresenceCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchPresenceSettingsCallback = std::function<void(TTV_ErrorCode ec, const PresenceSettings& settings)>;
  using SetPresenceSettingsCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchFriendListCallback = std::function<void(TTV_ErrorCode ec, const std::vector<Friend>& friends)>;
  using UpdateFriendshipCallback =
    std::function<void(TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status)>;
  using FetchFriendRequestsCallback = std::function<void(TTV_ErrorCode ec, const std::vector<FriendRequest>& requests)>;
  using FetchUnreadFriendRequestCountCallback = std::function<void(TTV_ErrorCode ec, uint32_t requestCount)>;
  using MarkAllFriendRequestsReadCallback = std::function<void(TTV_ErrorCode ec)>;
  using FetchFriendStatusCallback = std::function<void(TTV_ErrorCode ec, FriendStatus status)>;
  using FetchRecommendedFriendsCallback =
    std::function<void(TTV_ErrorCode ec, const std::vector<UserInfo>& recommendedFriends)>;
  using DismissRecommendedFriendCallback = std::function<void(TTV_ErrorCode ec)>;

  SocialAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);
  TTV_ErrorCode SetListener(const std::shared_ptr<ISocialAPIListener>& listener);
  TTV_ErrorCode SetEnabledFeatures(const FeatureFlags& features);

  /**
   * Manually pings the backend presence service to indicate that the client is still alive.  This normally does not
   * need to be done since presence is automatically maintained internally.  Use this only in circumstances where the
   * client was offline for a period of time.  The result will be returned via the passed-in callback.
   */
  TTV_ErrorCode PostPresence(UserId userId, const PostPresenceCallback& callback);

  /**
   * Sets the presence availability mode for the given user.
   */
  TTV_ErrorCode SetPresenceSessionAvailability(UserId userId, PresenceSessionAvailability availability);

  /**
   * Adds a watching activity to the user's list of activities.
   * channelId is the id of the channel being watched.
   * Returns a unique token that corresponds to that activity.
   */
  TTV_ErrorCode AddWatchingActivity(UserId userId, ChannelId channelId, PresenceActivityToken& activityToken);

  /**
   * Adds a playing activity to the user's list of activities.
   * gameId is the id of the game being played.
   * gameDisplayContext is an optional string that specifies additional context about the game.
   * Returns a unique token that corresponds to that activity.
   */
  TTV_ErrorCode AddPlayingActivity(
    UserId userId, GameId gameId, const std::string& gameDisplayContext, PresenceActivityToken& activityToken);

  /**
   * Removes the activity that corresponds to the passed in activity token.
   */
  TTV_ErrorCode RemoveActivity(UserId userId, PresenceActivityToken activityToken);

  /**
   * Initiates an async retrieval of presence settings from the backend.  The result will be returned via the passed-in
   * callback.
   */
  TTV_ErrorCode FetchPresenceSettings(UserId userId, const FetchPresenceSettingsCallback& callback);

  /**
   * Applies the given presence settings.
   *
   * @param[in] userId The id of the user we're changing the presence settings of.
   * @param[in] settings Settings include a user's presence override and whether they're sharing their current activity.
   *      A presence override means that the user will appear as that presence availability to their friends regardless
   * of the states of their sessions. If a user sets their presence as UserAvailability::None, then their sessions
   * determine their availability.
   * @param[in] callback Callback that returns the result PresenceSettings.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the request to update the presence settings.
   *   - TTV_EC_SHUT_DOWN: Social API is not initialized.
   *   - TTV_EC_REQUEST_PENDING: There's already a request to set the presence settings.
   *   - TTV_EC_NEED_TO_LOGIN: The user is not logged in.
   */
  TTV_ErrorCode SetPresenceSettings(
    UserId userId, const PresenceSettings& settings, const SetPresenceSettingsCallback& callback);

  /**
   * Specifies whether or not the presence system should automatically posting the user's presence.
   */
  TTV_ErrorCode SetAutomaticPresencePostingEnabled(UserId userId, bool enabled);

  /**
   * Determines whether or not the presence system is automatically posting the user's presence.
   */
  TTV_ErrorCode GetAutomaticPresencePostingEnabled(UserId userId, bool& enabled);

  /**
   * Initiates an async retrieval of the friend list.  The result will be notified via
   * ISocialAPIListener::SocialFetchFriendListComplete.
   */
  TTV_ErrorCode FetchFriendList(UserId userId, const FetchFriendListCallback& callback);

  /**
   * Initiates an async change of status with a friend.  This is used to send, accept and reject friend requests and
   * remove an existing friend. The result will be notified via ISocialAPIListener::SocialUpdateFriendshipComplete.
   */
  TTV_ErrorCode UpdateFriendship(
    UserId userId, UserId otherUserId, FriendAction action, const UpdateFriendshipCallback& callback);

  /**
   * Initiates an async retrieval of outstanding friend requests.  The result will be notified via
   * ISocialAPIListener::SocialFetchFriendRequestsComplete.
   */
  TTV_ErrorCode FetchFriendRequests(UserId userId, const FetchFriendRequestsCallback& callback);

  /**
   * Initiates an async retrieval of the number of unread friend requests.  The result will be notified via
   * ISocialAPIListener::SocialFetchUnreadFriendRequestsCountComplete.
   */
  TTV_ErrorCode FetchUnreadFriendRequestCount(UserId userId, const FetchUnreadFriendRequestCountCallback& callback);

  /**
   * Asynchronously marks all friend requests as read.  The result will be notified via
   * ISocialAPIListener::SocialMarkAllFriendRequestsReadComplete. NOTE: This does not reject any requsts.
   */
  TTV_ErrorCode MarkAllFriendRequestsRead(UserId userId, const MarkAllFriendRequestsReadCallback& callback);

  /**
   * Initiates an async retrieval of the relationship between two users.  The result will be notified via the passed-in
   * callback. This does not update any internal cache.
   */
  TTV_ErrorCode FetchFriendStatus(UserId userId, UserId otherUserId, const FetchFriendStatusCallback& callback);

  /**
   * Initiates an async retrieval of recommended friends.  The result will be notified via
   * ISocialAPIListener::SocialFetchRecommendedFriendsComplete.
   */
  TTV_ErrorCode FetchRecommendedFriends(UserId userId, const FetchRecommendedFriendsCallback& callback);

  /**
   * Initiates an async dimissal of the named user from the recommended friend list.  The result will be notified via
   * ISocialAPIListener::SocialDismissRecommendedFriendComplete.
   */
  TTV_ErrorCode DismissRecommendedFriend(
    UserId userId, UserId dismissUserId, const DismissRecommendedFriendCallback& callback);

 private:
  class CoreApiClient : public ICoreApiClient {
   public:
    CoreApiClient(SocialAPI* api);

    // ICoreApiClient implementation
    virtual std::string GetClientName();
    virtual void GetRequiredOAuthScopes(std::vector<std::string>& scopes);
    virtual void CoreUserLoggedIn(std::shared_ptr<User> user);
    virtual void CoreUserLoggedOut(std::shared_ptr<User> user);

   private:
    SocialAPI* mApi;
  };

  // ICoreApiClient implementation
  void CoreUserLoggedIn(std::shared_ptr<User> user);
  void CoreUserLoggedOut(std::shared_ptr<User> user);

  TTV_ErrorCode GetPresenceForUser(UserId userId, std::shared_ptr<Presence>& presence);
  TTV_ErrorCode GetFriendListForUser(UserId userId, std::shared_ptr<FriendList>& friendList);

  void CreateFriendListListener();
  void CreatePresenceListener();

  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  std::shared_ptr<CoreAPI> mCoreApi;
  std::shared_ptr<ICoreApiClient> mCoreApiClient;

  std::shared_ptr<SocialAPIInternalData> mInternalData;
  std::shared_ptr<UserRepository> mUserRepository;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;

  FeatureFlags mEnabledFeatures;
};
