/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/social/socialerrortypes.h"

namespace ttv {
namespace social {
struct FeatureFlags {
  FeatureFlags();

  static FeatureFlags All();
  static FeatureFlags None();

  bool friendList : 1;
  bool friendRequests : 1;
  bool presence : 1;
};

/**
 * Represents a user's current availability.
 * "Idle" is a deprecated user availability and will be parsed as Away.
 */
enum class PresenceUserAvailability { Offline, Online, Away, Busy };

/**
 * Represents a user's specific session's availability.
 */
enum class PresenceSessionAvailability { Offline, Online, Idle };

/**
 * Settings that describe the client's desired presence posting behavior.
 * The server exposes and synchronizes these settings between clients so that
 * the clients know how they should post presence.
 */
struct PresenceSettings {
  PresenceSettings();

  bool operator==(const PresenceSettings& rhs);
  bool operator!=(const PresenceSettings& rhs);

  /**
   * Represents possible values a user can override their availability with through PresenceSettings
   */
  enum class AvailabilityOverride { None, Offline, Away, Busy };

  /**
   * A user can explicitly set how they want their availability to appear to their friends, ignoring the state of their
   * sessions. If a user sets their presence as AvailabilityOverride::None, that is considered to be no override,
   * causing their sessions to determine their availability.
   */
  AvailabilityOverride availabilityOverride;

  /**
   * Whether the client should include their current activity when posting presence.
   */
  bool shareActivity : 1;
};

/**
 * A unique token that is associated with an added presence activity.
 * This is vended by the API when adding an activity, and the client
 * passes this in when removing the activity.
 */
using PresenceActivityToken = uint32_t;

/**
 * Abstract base class for an activity.
 */
struct PresenceActivity {
  /**
   * Enumeration that maps to each possible subclass of PresenceActivity.
   */
  enum class Type { Broadcasting, Watching, Playing, Unknown };

  virtual ~PresenceActivity() = default;

  PresenceActivity() = default;
  PresenceActivity(const PresenceActivity& src) = default;
  PresenceActivity& operator=(const PresenceActivity& src) = default;
  PresenceActivity(PresenceActivity&& src) = default;
  PresenceActivity& operator=(PresenceActivity&& src) = default;

  /**
   * The return value indicates which concrete activity subclass this object is.
   */
  virtual Type GetType() const = 0;

  /**
   * Makes a safe copy of the activity of the matching subclass, without slicing.
   */
  virtual std::unique_ptr<PresenceActivity> Clone() const = 0;
};

/**
 * Activity indicating the user is streaming.
 */
struct BroadcastingActivity : public Cloneable<PresenceActivity, BroadcastingActivity> {
  BroadcastingActivity();

  std::string channelLogin;        //!< The login of the channel the user is broadcasting from.
  std::string channelDisplayName;  //!< The display name of the channel the user is broadcasting from.
  std::string gameName;            //!< The name of the game being played on the channel.

  ChannelId channelId;  //!< The channelId of the channel the user is broadcasting from.
  GameId gameId;        //!< The ID of the game being played on the channel.

  virtual Type GetType() const override { return Type::Broadcasting; }
};

/**
 * Activity indicating the user is watching a stream.
 */
struct WatchingActivity : public Cloneable<PresenceActivity, WatchingActivity> {
  WatchingActivity();

  std::string channelLogin;        //!< The login of the channel being viewed.
  std::string channelDisplayName;  //!< The display name of the channel being viewed.

  std::string
    hostedChannelLogin;  //!< The login of a channel that is being hosted by the viewed channel, or empty string if it is not hosting.
  std::string
    hostedChannelDisplayName;  //!< The display name of the channel that is being hosted by the viewed channel, or empty string if it is not hosting.

  std::string gameName;  //!< The name of the game being played on the channel.

  ChannelId channelId;  //!< The channelId of the channel being viewed.
  ChannelId
    hostedChannelId;  //!< The channelID of the channel being hosted by the viewed channel, or 0 if it is not hosting.
  GameId gameId;      //!< The ID of the game being played on the channel.

  virtual Type GetType() const override { return Type::Watching; }
};

/**
 * Activity indicating the user is playing a game.
 */
struct PlayingActivity : public Cloneable<PresenceActivity, PlayingActivity> {
  PlayingActivity();

  std::string gameName;  //!< The name of the game being played.
  std::string
    gameDisplayContext;  //!< Some game specific context string that may indicate some extra info about the state of the game being played.

  GameId gameId;  //!< The ID of the game being played.

  virtual Type GetType() const override { return Type::Playing; }
};

/**
 * Information about the user's current presence.
 */
struct PresenceStatus {
  PresenceStatus();
  PresenceStatus(const PresenceStatus& source);
  PresenceStatus(PresenceStatus&& source) = default;
  PresenceStatus& operator=(const PresenceStatus& source);
  PresenceStatus& operator=(PresenceStatus&& source) = default;

  std::unique_ptr<PresenceActivity>
    activity;  //!< The activity the user is engaging in, or nullptr if the user is not sharing any activity.
  PresenceUserAvailability availability;  //!< The offline/online presence of the user.
  Timestamp lastUpdate;                   //!< The last time this user's presence changed.
};

/**
 * The type of action to take when updating friendship.
 */
enum class FriendAction { SendRequest, AcceptRequest, RejectRequest, DeleteFriend };

/**
 * The outcome of a given friendship update.
 */
enum class UpdateFriendResult {
  /**
   * In response to sending a request. The request was successfully sent.
   */
  RequestSent,

  /**
   * In response to accepting a request. The request was successfuly accepted.
   */
  RequestAccepted,

  /**
   * In response to sending a request. Request could not be sent, due to one of the following:
   * 1) The requesting user has too many friend requests in flight.
   * 2) The requesting user has a full friends list.
   * 3) The receiving user has blocked the requesting user.
   */
  RequestNotAllowed,

  /**
   * In response to accepting or rejecting a request. The request itself does not exist.
   */
  RequestNotFound,

  /**
   * In response to rejecting a request. The request was successfully rejected.
   */
  RequestRejected,

  /**
   * In response to deleting a request. The request was successfully deleted.
   */
  RequestDeleted,

  /**
   * In response to deleting a friend. The friend was successfully deleted.
   */
  FriendDeleted,

  /**
   * In response to deleting a friend. The friend was not found in the deleting user's friend list.
   */
  FriendNotFound,

  /**
   * In response to sending a friend request. The target user is already in the sender's friend list.
   */
  AlreadyExists,

  /**
   * Some other unkown response.
   */
  Unknown
};

enum class FriendRequestRemovalReason {
  Invalid,         //!< The request doesn't exist.
  SelfAccepted,    //!< The local user accepted.
  TargetAccepted,  //!< The other user accepted.
  SelfRejected,    //!< The local user rejected.
  TargetRejected   //!< The other user rejected.
};

enum class FriendStatus {
  /**
   * The users have no friendship, no blocking relationship, and no pending friend requests in either direction.
   */
  NoRelation,

  /**
   * The target user is blocked by the client user.
   */
  Blocked,

  /**
   * The client user is blocked by the target user.
   */
  Blocks,

  /**
   * The client user has a pending friend request out to the target user.
   */
  SentRequest,

  /**
   * The target user has a pending friend request out to the client user.
   */
  ReceivedRequest,

  /**
   * The users are in each others' friends list.
   */
  Friends,

  /**
   * Some other unknown state.
   */
  Unknown
};

/**
 * An entry in the users' friends list.
 */
struct Friend {
  Friend();

  /**
   * Data about the user.
   */
  UserInfo userInfo;

  /**
   * When the user became friends, measured as seconds since Unix epoch.
   */
  Timestamp friendsSinceTime;

  /**
   * The user's current presence status.
   */
  PresenceStatus presenceStatus;
};

/**
 * A pending friend request.
 */
struct FriendRequest {
  FriendRequest();

  /**
   * Data about the user requesting or being requested.
   */
  UserInfo userInfo;

  /**
   * When the request was issues, measured as seconds since Unix epoch.
   */
  Timestamp requestTime;
};
}  // namespace social
}  // namespace ttv
