#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/social/socialapi.h"
#include "twitchsdk/social/sociallistener.h"

using namespace ttv;
using namespace ttv::social;

namespace ttv {
std::string ToString(FriendAction action) {
  switch (action) {
    case FriendAction::SendRequest:
      return "SendRequest";
    case FriendAction::AcceptRequest:
      return "AcceptRequest";
    case FriendAction::RejectRequest:
      return "RejectRequest";
    case FriendAction::DeleteFriend:
      return "DeleteFriend";
    default:
      return "???";
  }
}

std::string ToString(UpdateFriendResult result) {
  switch (result) {
    case UpdateFriendResult::Unknown:
      return "Unknown";
    case UpdateFriendResult::RequestSent:
      return "RequestSent";
    case UpdateFriendResult::RequestAccepted:
      return "RequestAccepted";
    case UpdateFriendResult::RequestNotAllowed:
      return "RequestNotAllowed";
    case UpdateFriendResult::RequestNotFound:
      return "RequestNotFound";
    case UpdateFriendResult::RequestRejected:
      return "RequestRejected";
    case UpdateFriendResult::RequestDeleted:
      return "RequestDeleted";
    case UpdateFriendResult::FriendDeleted:
      return "FriendDeleted";
    case UpdateFriendResult::FriendNotFound:
      return "FriendNotFound";
    case UpdateFriendResult::AlreadyExists:
      return "AlreadyExists";
    default:
      return "???";
  }
}

std::string ToString(FriendRequestRemovalReason reason) {
  switch (reason) {
    case FriendRequestRemovalReason::Invalid:
      return "Invalid";
    case FriendRequestRemovalReason::SelfAccepted:
      return "SelfAccepted";
    case FriendRequestRemovalReason::TargetAccepted:
      return "TargetAccepted";
    case FriendRequestRemovalReason::SelfRejected:
      return "SelfRejected";
    case FriendRequestRemovalReason::TargetRejected:
      return "TargetRejected";
    default:
      return "???";
  }
}

std::string ToString(FriendStatus status) {
  switch (status) {
    case FriendStatus::Unknown:
      return "Unknown";
    case FriendStatus::NoRelation:
      return "NoRelation";
    case FriendStatus::Blocked:
      return "Blocked";
    case FriendStatus::Blocks:
      return "Blocks";
    case FriendStatus::SentRequest:
      return "SentRequest";
    case FriendStatus::ReceivedRequest:
      return "ReceivedRequest";
    case FriendStatus::Friends:
      return "Friends";
    default:
      return "???";
  }
}

std::string ToString(PresenceUserAvailability availability) {
  switch (availability) {
    case PresenceUserAvailability::Online:
      return "Online";
    case PresenceUserAvailability::Offline:
      return "Offline";
    case PresenceUserAvailability::Away:
      return "Away";
    case PresenceUserAvailability::Busy:
      return "Busy";
    default:
      return "???";
  }
}

std::string ToString(PresenceSettings::AvailabilityOverride availability) {
  switch (availability) {
    case PresenceSettings::AvailabilityOverride::None:
      return "None";
    case PresenceSettings::AvailabilityOverride::Offline:
      return "Offline";
    case PresenceSettings::AvailabilityOverride::Away:
      return "Away";
    case PresenceSettings::AvailabilityOverride::Busy:
      return "Busy";
    default:
      return "???";
  }
}

void Print(const PresenceActivity& activity, std::string indent) {
  switch (activity.GetType()) {
    case PresenceActivity::Type::Broadcasting: {
      std::cout << indent << "type: Broadcasting" << std::endl;

      const auto& broadcastingActivity = static_cast<const BroadcastingActivity&>(activity);

      std::cout << indent << PRINT_FIELD(broadcastingActivity, channelId) << std::endl;
      std::cout << indent << PRINT_FIELD(broadcastingActivity, channelLogin) << std::endl;
      std::cout << indent << PRINT_FIELD(broadcastingActivity, channelDisplayName) << std::endl;

      std::cout << indent << PRINT_FIELD(broadcastingActivity, gameId) << std::endl;
      std::cout << indent << PRINT_FIELD(broadcastingActivity, gameName) << std::endl;
      break;
    }
    case PresenceActivity::Type::Playing: {
      std::cout << indent << "type: Playing" << std::endl;

      const auto& playingActivity = static_cast<const PlayingActivity&>(activity);

      std::cout << indent << PRINT_FIELD(playingActivity, gameId) << std::endl;
      std::cout << indent << PRINT_FIELD(playingActivity, gameName) << std::endl;
      std::cout << indent << PRINT_FIELD(playingActivity, gameDisplayContext) << std::endl;
      break;
    }
    case PresenceActivity::Type::Watching: {
      std::cout << indent << "type: Watching" << std::endl;

      const auto& watchingActivity = static_cast<const WatchingActivity&>(activity);

      std::cout << indent << PRINT_FIELD(watchingActivity, channelId) << std::endl;
      std::cout << indent << PRINT_FIELD(watchingActivity, channelLogin) << std::endl;
      std::cout << indent << PRINT_FIELD(watchingActivity, channelDisplayName) << std::endl;

      std::cout << indent << PRINT_FIELD(watchingActivity, hostedChannelId) << std::endl;
      std::cout << indent << PRINT_FIELD(watchingActivity, hostedChannelLogin) << std::endl;
      std::cout << indent << PRINT_FIELD(watchingActivity, hostedChannelDisplayName) << std::endl;

      std::cout << indent << PRINT_FIELD(watchingActivity, gameId) << std::endl;
      std::cout << indent << PRINT_FIELD(watchingActivity, gameName) << std::endl;
      break;
    }
    default: {
      std::cout << indent << "type: None" << std::endl;
      break;
    }
  }
}

void Print(const PresenceStatus& presence, std::string indent) {
  std::cout << indent << "Presence: " << std::endl;
  indent += "  ";

  std::cout << indent << PRINT_TIMESTAMP_FIELD(presence, lastUpdate) << std::endl;
  std::cout << indent << "availability:" << ToString(presence.availability) << std::endl;
  if (presence.activity != nullptr) {
    std::cout << indent << "activity:" << std::endl;
    Print(*presence.activity, indent + "  ");
  }
}

void Print(const Friend& entry, std::string indent) {
  std::cout << indent << "Friend: " << entry.userInfo.userId << std::endl;
  indent += "  ";

  std::cout << indent << "userInfo:" << std::endl;
  Print(entry.userInfo, indent + "  ");

  std::cout << indent << PRINT_TIMESTAMP_FIELD(entry, friendsSinceTime) << std::endl;

  std::cout << indent << "presence:" << std::endl;
  Print(entry.presenceStatus, indent + "  ");
}

void Print(const std::vector<Friend>& list, const std::string& indent) {
  std::cout << indent << "Friends (" << list.size() << ")" << std::endl;
  for (const auto& friendEntry : list) {
    Print(friendEntry, indent + "  ");
  }
}

void Print(const FriendRequest& request, std::string indent) {
  std::cout << indent << "Friend request: " << request.userInfo.userId << std::endl;
  indent += "  ";

  std::cout << indent << "userInfo:" << std::endl;
  Print(request.userInfo, indent + "  ");

  std::cout << indent << PRINT_TIMESTAMP_FIELD(request, requestTime) << std::endl;
}

void Print(const std::vector<FriendRequest>& list, const std::string& indent) {
  std::cout << indent << "Friend requests (" << list.size() << ")" << std::endl;
  for (const auto& request : list) {
    Print(request, indent + "  ");
  }
}
}  // namespace ttv

namespace {
const char* kTraceComponents[] = {"SocialAPI", "SocialFriendRequestsTask", "SocialGetFriendsPresenceTask",
  "SocialGetFriendsTask", "SocialPresenceSettingsTask", "SocialUpdateFriendTask", "SocialGetUserInfoTask", "FriendList",
  "Presence"};

class SocialApiListener;

std::shared_ptr<SocialAPI> gSocialApi;
std::shared_ptr<SocialApiListener> gSocialApiListener;

std::map<std::string, std::map<PresenceActivityToken, PresenceActivity::Type>> gAddedPresencesMap;

class SocialApiListener : public ttv::social::ISocialAPIListener {
 public:
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) override {
    std::cout << "ModuleStateChanged: " << ::ToString(state) << " " << ErrorToString(ec) << std::endl;
  }

  virtual void SocialFriendInfoChanged(UserId userId, const std::vector<Friend>& changes) override {
    std::cout << "SocialFriendInfoChanged: " << userId << std::endl;

    Print(changes, "  ");
  }

  virtual void SocialFriendshipChanged(
    UserId userId, const std::vector<Friend>& added, const std::vector<Friend>& removed) override {
    std::cout << "SocialFriendshipChanged: " << userId << std::endl;

    std::cout << "  added:" << std::endl;
    Print(added, "    ");

    std::cout << "  removed:" << std::endl;
    Print(removed, "    ");
  }

  virtual void SocialRealtimeFriendRequestReceived(UserId userId, const FriendRequest& request) override {
    std::cout << "SocialRealtimeFriendRequestsReceived: " << userId << std::endl;
    Print(request, "  ");
  }

  virtual void SocialUnreadFriendRequestCountChanged(UserId userId, uint32_t count) override {
    std::cout << "SocialUnreadFriendRequestCountChanged: " << userId << " " << count << std::endl;
  }

  virtual void SocialFriendRequestRemoved(
    UserId userId, UserId otherUserId, FriendRequestRemovalReason reason) override {
    std::cout << "SocialFriendRequestRemoved: " << userId << " " << otherUserId << " " << ToString(reason) << std::endl;
  }

  virtual void SocialPresenceSettingsChanged(UserId userId, const PresenceSettings& settings) override {
    std::cout << "SocialPresenceSettingsChanged: " << userId
              << " availibilityOverride: " << ToString(settings.availabilityOverride)
              << " shareActivity: " << settings.shareActivity << std::endl;
  }
};

std::shared_ptr<CommandCategory> RegisterSocialCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Social", "The ttv::social::SocialAPI command interface");

  category->AddCommand("SocialInit")
    .AddFunction()
    .Description("Calls SocialAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      // TODO: Maybe make these properties to be settable
      gSocialApi->SetEnabledFeatures(FeatureFlags::All());
      gSocialApi->SetListener(gSocialApiListener);

      TTV_ErrorCode ec = gSocialApi->Initialize([](TTV_ErrorCode ec) {
        std::cout << "SocialInit completed with error";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialGetState")
    .AddFunction()
    .Description("Calls SocialAPI::GetState")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      IModule::State state = gSocialApi->GetState();
      std::cout << "  " << ::ToString(state) << std::endl;
      return true;
    });

  category->AddCommand("SocialShutdown")
    .AddFunction()
    .Description("Calls SocialAPI::Shutdown")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gSocialApi->Shutdown([](TTV_ErrorCode ec) {
        std::cout << "SocialShutdown completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);

      if (TTV_SUCCEEDED(ec)) {
        gSocialApiListener.reset();
      }

      return true;
    });

  category->AddCommand("SocialSetPresenceSessionAvailability")
    .AddAlias("SetPresenceAvailability")
    .AddFunction()
    .Description("Calls SocialAPI::SetPresenceSessionAvailability")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("availability", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      PresenceSessionAvailability availability;

      if (params[1].GetString() == "online") {
        availability = PresenceSessionAvailability::Online;
      } else if (params[1].GetString() == "offline") {
        availability = PresenceSessionAvailability::Offline;
      } else if (params[1].GetString() == "idle") {
        availability = PresenceSessionAvailability::Idle;
      } else {
        ReportCommandResult(TTV_EC_INVALID_ARG);
        return true;
      }

      TTV_ErrorCode ec = gSocialApi->SetPresenceSessionAvailability(params[0], availability);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialPostPresence")
    .AddAlias("PostPresence")
    .AddFunction()
    .Description("Calls SocialAPI::PostPresence")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->PostPresence(params[0], [](TTV_ErrorCode ec) {
        std::cout << "PostPresence completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialAddWatchingActivity")
    .AddAlias("AddWatchingActivity")
    .AddFunction()
    .Description("Calls SocialAPI::AddActivity with a watching activity")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("channelId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) {
      uint32_t activityToken;
      TTV_ErrorCode ec = gSocialApi->AddWatchingActivity(params[0], params[1], activityToken);
      ReportCommandResult(ec);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  activity token: " << activityToken << std::endl;
        gAddedPresencesMap[params[0]].emplace(activityToken, PresenceActivity::Type::Watching);
      }
      return true;
    });

  category->AddCommand("SocialAddPlayingActivity")
    .AddAlias("AddPlayingActivity")
    .AddFunction()
    .Description("Calls SocialAPI::AddPlayingActivity")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("gameId", ParamType::UInt32)
    .AddParam("context", ParamType::String)
    .Function([](const std::vector<ParamValue>& params) {
      uint32_t activityToken;
      TTV_ErrorCode ec = gSocialApi->AddPlayingActivity(params[0], params[1], params[2], activityToken);
      ReportCommandResult(ec);
      if (TTV_SUCCEEDED(ec)) {
        std::cout << "  activity token: " << activityToken << std::endl;
        gAddedPresencesMap[params[0]].emplace(activityToken, PresenceActivity::Type::Playing);
      }
      return true;
    });

  category->AddCommand("SocialRemoveActivity")
    .AddAlias("RemoveActivity")
    .AddFunction()
    .Description("Calls SocialAPI::RemoveActivity")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("activityToken", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) {
      TTV_ErrorCode ec = gSocialApi->RemoveActivity(params[0], params[1]);
      ReportCommandResult(ec);
      if (TTV_SUCCEEDED(ec)) {
        auto userMapIter = gAddedPresencesMap.find(params[0]);
        if (userMapIter != gAddedPresencesMap.end()) {
          auto& activityMap = userMapIter->second;
          auto activityMapIter = activityMap.find(params[1]);
          if (activityMapIter != activityMap.end()) {
            activityMap.erase(activityMapIter);
          }
        }
      }
      return true;
    });

  category->AddCommand("SocialListActivities")
    .AddAlias("ListActivities")
    .AddFunction()
    .Description("Lists the activities added in this session")
    .Function([](const std::vector<ParamValue>& /*params*/) {
      std::cout << "Activities:" << std::endl;
      for (const auto& userMapIter : gAddedPresencesMap) {
        std::cout << "  User:" << userMapIter.first << std::endl;
        for (const auto& activityMapIter : userMapIter.second) {
          std::string typeString;
          switch (activityMapIter.second) {
            case PresenceActivity::Type::Playing: {
              typeString = "Playing";
              break;
            }
            case PresenceActivity::Type::Watching: {
              typeString = "Watching";
              break;
            }
            default: {
              typeString = "Unknown";
              break;
            }
          }
          std::cout << "    Token: " << activityMapIter.first << " Type: " << typeString << std::endl;
        }
      }
      return true;
    });

  category->AddCommand("SocialFetchPresenceSettings")
    .AddAlias("FetchPresenceSettings")
    .AddFunction()
    .Description("Calls SocialAPI::FetchPresenceSettings")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gSocialApi->FetchPresenceSettings(params[0], [](TTV_ErrorCode ec, const PresenceSettings& settings) {
          std::cout << "FetchPresenceSettings completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Settings: availabilityOverride: " << ToString(settings.availabilityOverride)
                      << " shareActivity: " << settings.shareActivity << std::endl;
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialSetPresenceSettings")
    .AddAlias("SetPresenceSettings")
    .AddFunction()
    .Description("Calls SocialAPI::SetPresenceSettings")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("availabilityOverride", ParamType::String)
    .AddParam("shareActivity", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      PresenceSettings settings;

      if (params[1].GetString() == "none") {
        settings.availabilityOverride = PresenceSettings::AvailabilityOverride::None;
      } else if (params[1].GetString() == "offline") {
        settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Offline;
      } else if (params[1].GetString() == "away") {
        settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Away;
      } else if (params[1].GetString() == "busy") {
        settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Busy;
      }

      settings.shareActivity = params[2];

      TTV_ErrorCode ec = gSocialApi->SetPresenceSettings(params[0], settings, [](TTV_ErrorCode ec) {
        std::cout << "SetPresenceSettings completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialSetAutomaticPresencePostingEnabled")
    .AddAlias("SetAutomaticPresencePostingEnabled")
    .AddAlias("enableautopost")
    .AddFunction()
    .Description("Calls SocialAPI::SetAutomaticPresencePostingEnabled")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("enable", ParamType::Bool)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->SetAutomaticPresencePostingEnabled(params[0], params[1]);
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialFetchFriendList")
    .AddAlias("FetchFriendList")
    .AddFunction()
    .Description("Calls SocialAPI::FetchFriendList")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gSocialApi->FetchFriendList(params[0], [](TTV_ErrorCode ec, const std::vector<Friend>& friends) {
          std::cout << "FetchFriendList completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            Print(friends, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialSendFriendRequest")
    .AddAlias("SendFriendRequest")
    .AddFunction()
    .Description("Calls SocialAPI::UpdateFriendship to send a friend request")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->UpdateFriendship(params[0], params[1], FriendAction::SendRequest,
        [](TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status) {
          std::cout << "SendFriendRequest completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Result: " << ToString(result);
            std::cout << "  FriendStatus: " << ToString(status);
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialAcceptFriendRequest")
    .AddAlias("AcceptFriendRequest")
    .AddFunction()
    .Description("Calls SocialAPI::UpdateFriendship to accpet a friend request")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->UpdateFriendship(params[0], params[1], FriendAction::AcceptRequest,
        [](TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status) {
          std::cout << "AcceptFriendRequest completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Result: " << ToString(result);
            std::cout << "  FriendStatus: " << ToString(status);
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialRejectFriendRequest")
    .AddAlias("RejectFriendRequest")
    .AddFunction()
    .Description("Calls SocialAPI::UpdateFriendship to reject a friend request")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->UpdateFriendship(params[0], params[1], FriendAction::RejectRequest,
        [](TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status) {
          std::cout << "RejectFriendRequest completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Result: " << ToString(result);
            std::cout << "  FriendStatus: " << ToString(status);
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialRemoveFriend")
    .AddAlias("RemoveFriend")
    .AddFunction()
    .Description("Calls SocialAPI::UpdateFriendship to remove a friend")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("otherUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->UpdateFriendship(params[0], params[1], FriendAction::DeleteFriend,
        [](TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status) {
          std::cout << "RemoveFriend completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Result: " << ToString(result);
            std::cout << "  FriendStatus: " << ToString(status);
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialFetchFriendRequests")
    .AddAlias("FetchFriendRequests")
    .AddFunction()
    .Description("Calls SocialAPI::FetchFriendRequests")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec =
        gSocialApi->FetchFriendRequests(params[0], [](TTV_ErrorCode ec, const std::vector<FriendRequest>& requests) {
          std::cout << "FetchFriendRequests completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            Print(requests, "  ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialFetchUnreadFriendRequestCount")
    .AddAlias("FetchUnreadFriendRequestCount")
    .AddFunction()
    .Description("Calls SocialAPI::FetchUnreadFriendRequestCount")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->FetchUnreadFriendRequestCount(params[0], [](TTV_ErrorCode ec, uint32_t count) {
        std::cout << "FetchUnreadFriendRequestCount completed with error:";
        ReportCommandResult(ec);
        if (TTV_SUCCEEDED(ec)) {
          std::cout << "  Count: " << count << std::endl;
        }
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialMarkAllFriendRequestsRead")
    .AddAlias("MarkAllFriendRequestsRead")
    .AddFunction()
    .Description("Calls SocialAPI::MarkAllFriendRequestsRead")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->MarkAllFriendRequestsRead(params[0], [](TTV_ErrorCode ec) {
        std::cout << "MarkAllFriendRequestsRead completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialFetchRecommendedFriends")
    .AddAlias("FetchRecommendedFriends")
    .AddFunction()
    .Description("Calls SocialAPI::FetchRecommendedFriends")
    .AddParam("userId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->FetchRecommendedFriends(
        params[0], [](TTV_ErrorCode ec, const std::vector<UserInfo>& recommendedFriends) {
          std::cout << "FetchRecommendedFriends completed with error:";
          ReportCommandResult(ec);
          if (TTV_SUCCEEDED(ec)) {
            std::cout << "  Recommendations:" << std::endl;
            Print(recommendedFriends, "    ");
          }
        });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("SocialDismissRecommendedFriend")
    .AddAlias("DismissRecommendedFriend")
    .AddFunction()
    .Description("Calls SocialAPI::DismissRecommendedFriend")
    .AddParam("userId", ParamType::UInt32)
    .AddParam("dismissUserId", ParamType::UInt32)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      TTV_ErrorCode ec = gSocialApi->DismissRecommendedFriend(params[0], params[1], [](TTV_ErrorCode ec) {
        std::cout << "DismissRecommendedFriend completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("PrintJavaSocialErrorCodes")
    .AddFunction()
    .Description("Dumps the Java snippet for updated values in SocialErrorCode.java")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      std::vector<EnumValue> values;
      ttv::social::GetSocialErrorCodeValues(values);

      for (const auto& v : values) {
        std::cout << "public static final ErrorCode " << v.name << " = new SocialErrorCode(0x" << std::hex << v.value
                  << ", \"" << v.name << "\");" << std::endl;
      }

      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));

      gSocialApi = std::make_shared<SocialAPI>();
      gSocialApi->SetCoreApi(coreApi);
      context->RegisterModule(gSocialApi);

      gSocialApiListener = std::make_shared<SocialApiListener>();

      auto commands = RegisterSocialCommands(context);
      context->RegisterCommandCategory(commands);
    },
    1);
});
}  // namespace
