/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/friendlist.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/lambdatask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/social/internal/socialhelpers.h"
#include "twitchsdk/social/internal/task/socialfriendrequeststask.h"
#include "twitchsdk/social/internal/task/socialgetfriendspresencetask.h"
#include "twitchsdk/social/internal/task/socialrecommendedfriendstask.h"

#include <ctime>

namespace {
const char* kLogger = "FriendList";

uint64_t kFailedRetryIntervalMilliseconds = 1 * 60 * 1000;
uint64_t kFailedRetryJitterMilliseconds = 30 * 1000;
uint64_t kCoarseResyncIntervalMilliseconds = 10 * 60 * 1000;  // Full refresh every 10 minutes
uint64_t kInitialFriendListFetchDelayMilliseconds = 6 * 1000;
uint64_t kInitialFriendRequestFetchDelayMilliseconds = 6 * 1000;
uint64_t kInitialFriendRequestUnreadCountFetchDelayMilliseconds = 6 * 1000;
uint64_t kInitialRecommendedFriendsFetchDelayMilliseconds = 6 * 1000;
uint64_t kFriendAddedRefreshDelayMilliseconds = 5 * 1000;
}  // namespace

ttv::social::FriendList::FriendList(const std::shared_ptr<User>& user, const FeatureFlags& enabledFeatures)
    : UserComponent(user),
      mNumUnreadFriendRequests(0),
      mEnabledFeatures(enabledFeatures),
      mReceivedFirstFriendListFetch(false),
      mReceivedFirstFriendRequestsFetch(false),
      mReceivedFirstFriendRequestUnreadCountFetch(false),
      mReceivedFirstRecommendedFriendsFetch(false) {
  TTV_ASSERT(enabledFeatures.friendList);

  mFriendRequestFetcher = std::make_shared<PagedRequestFetcher>();
}

ttv::social::FriendList::~FriendList() {
  Log(MessageLevel::Debug, "FriendList - dtor");
}

void ttv::social::FriendList::AddListener(std::shared_ptr<IListener> listener) {
  mListeners.AddListener(listener);
}

void ttv::social::FriendList::RemoveListener(std::shared_ptr<IListener> listener) {
  mListeners.RemoveListener(listener);
}

TTV_ErrorCode ttv::social::FriendList::Initialize() {
  Log(MessageLevel::Debug, "FriendList::Initialize()");

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  TTV_ErrorCode ec = UserComponent::Initialize();
  if (TTV_FAILED(ec)) {
    return ec;
  }

  mPubSub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (mPubSub != nullptr) {
    mPubSubTopicListener = std::make_shared<PubSubTopicListener>(this);
    mPubSubTopicListenerHelper = std::make_shared<PubSubTopicListenerHelper>(mPubSub, mPubSubTopicListener);

    SubscribeTopics();
  }

  // Kick off fetching of friend and request lists
  mFriendListFetchTimer.Set(kInitialFriendListFetchDelayMilliseconds);
  mRecommendedFriendsFetchTimer.Set(kInitialRecommendedFriendsFetchDelayMilliseconds);

  if (mEnabledFeatures.friendRequests) {
    mFriendRequestFetchTimer.Set(kInitialFriendRequestFetchDelayMilliseconds);
    mFriendRequestUnreadCountFetchTimer.Set(kInitialFriendRequestUnreadCountFetchDelayMilliseconds);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::FriendList::SubscribeTopics() {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (mPubSubTopicListenerHelper == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  } else if (user->GetUserId() == 0) {
    Log(MessageLevel::Debug, "UserId not yet retrieved");

    return TTV_EC_USERINFO_NOT_AVAILABLE;
  } else if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  // Generate the topic ids we need to subscribe to
  char buffer[64];

  if (mPresenceTopic == "") {
    snprintf(buffer, sizeof(buffer), "presence.%u", user->GetUserId());
    mPresenceTopic = buffer;
  }

  if (mFriendshipTopic == "") {
    snprintf(buffer, sizeof(buffer), "friendship.%u", user->GetUserId());
    mFriendshipTopic = buffer;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (!mPubSubTopicListenerHelper->ContainsTopic(mPresenceTopic)) {
    ec = mPubSubTopicListenerHelper->Subscribe(mPresenceTopic);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  if (!mPubSubTopicListenerHelper->ContainsTopic(mFriendshipTopic)) {
    ec = mPubSubTopicListenerHelper->Subscribe(mFriendshipTopic);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  return ec;
}

void ttv::social::FriendList::AddToPendingFriendAdds(UserId userId) {
  // Add it to pending removes
  auto addIter = mPendingAddedFriends.find(userId);
  if (addIter == mPendingAddedFriends.end()) {
    mPendingAddedFriends.insert(userId);
  }

  // Remove it from pending adds if it's there
  auto removeIter = mPendingRemovedFriends.find(userId);
  if (removeIter != mPendingRemovedFriends.end()) {
    mPendingRemovedFriends.erase(removeIter);
  }
}

void ttv::social::FriendList::AddToPendingFriendRemoves(const FriendEntry& entry) {
  // Add it to pending removes
  auto userId = entry.friendData.userInfo.userId;
  auto removeIter = mPendingRemovedFriends.find(userId);
  if (removeIter == mPendingRemovedFriends.end()) {
    mPendingRemovedFriends[userId] = entry.friendData;
  }

  // Remove it from pending adds if it's there
  auto addIter = mPendingAddedFriends.find(userId);
  if (addIter != mPendingAddedFriends.end()) {
    mPendingAddedFriends.erase(addIter);
  }
}

TTV_ErrorCode ttv::social::FriendList::FetchFriendList(const FetchFriendListCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!mEnabledFeatures.friendList) {
    return TTV_EC_FEATURE_DISABLED;
  }

  mFetchFriendListCallbacks.Push(callback);

  // Kick off a fetch immediately on an explicit request.
  mFriendListFetchTimer.Set(0);

  return TTV_EC_SUCCESS;
}

void ttv::social::FriendList::RequestFriendsList() {
  if (mState != State::Initialized) {
    return;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  auto oauthToken = user->GetOAuthToken();

  mFriendListFetchTimer.Clear();

  auto fetchTask = std::make_shared<SocialGetFriendsPresenceTask>(user->GetUserId(), oauthToken->GetToken(),
    [this, user, oauthToken](SocialGetFriendsPresenceTask* source, TTV_ErrorCode ec,
      const std::shared_ptr<SocialGetFriendsPresenceTask::Result>& result) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec)) {
        TTV_ASSERT(result != nullptr);

        UserId userId = user->GetUserId();
        std::vector<FriendEntry> entries;
        for (const auto& presence : result->presenceList) {
          // This endpoint returns data about yourself, but we don't want our own user in the friends list, so filter it
          // out.
          if (presence.friendData.userInfo.userId != userId) {
            entries.emplace_back();
            auto& entry = entries.back();
            entry.friendData = presence.friendData;
            entry.presenceUpdateIndex = presence.presenceUpdateIndex;
          }
        }

        HandleFriendsList(std::move(entries));

        mFriendListFetchTimer.Set(kCoarseResyncIntervalMilliseconds);
      } else {
        if (ec == TTV_EC_AUTHENTICATION) {
          user->ReportOAuthTokenInvalid(oauthToken, ec);
        }

        // Try again soon
        mFriendListFetchTimer.SetWithJitter(kFailedRetryIntervalMilliseconds, kFailedRetryJitterMilliseconds);
      }
    });

  TTV_ErrorCode ec = StartTask(fetchTask);
  if (TTV_FAILED(ec)) {
    mFriendListFetchTimer.SetWithJitter(kFailedRetryIntervalMilliseconds, kFailedRetryJitterMilliseconds);
  }
}

void ttv::social::FriendList::SetFriendPresence(UserId userId, Timestamp updateTime, uint64_t index,
  PresenceUserAvailability availability, std::unique_ptr<PresenceActivity>&& activity) {
  // Update the presence entry
  auto iter = mFriendMap.find(userId);
  if ((iter == mFriendMap.end()) || (index <= iter->second.presenceUpdateIndex)) {
    // This is a stale update, or we don't have the info for the friend populated yet.
    return;
  }

  FriendEntry& entry = iter->second;
  entry.presenceUpdateIndex = index;

  PresenceStatus& status = entry.friendData.presenceStatus;
  // No change
  if (availability == status.availability && status.lastUpdate == updateTime) {
    return;
  }

  status.availability = availability;
  status.lastUpdate = updateTime;
  status.activity = std::move(activity);

  mPendingInfoChanges.insert(userId);
}

void ttv::social::FriendList::HandleFriendsList(std::vector<FriendEntry>&& list) {
  // Copy the current list
  std::map<UserId, FriendEntry> currentFriendMap;

  std::vector<UserId> addedFriends;
  std::vector<FriendEntry> removedFriends;

  // We don't want to mark the users we add the first time as new friends since it's the first fetch
  if (mReceivedFirstFriendListFetch) {
    auto previousFriendMap = mFriendMap;
    currentFriendMap = mFriendMap;

    // Update the friend list
    for (const auto& entry : list) {
      auto iter = previousFriendMap.find(entry.friendData.userInfo.userId);

      // New friend
      if (iter == previousFriendMap.end()) {
        addedFriends.push_back(entry.friendData.userInfo.userId);
      }
      // Existing friend
      else {
        previousFriendMap.erase(iter);
      }

      currentFriendMap[entry.friendData.userInfo.userId] = entry;
    }

    // Removed friends
    for (const auto& kvp : previousFriendMap) {
      UserId userId = kvp.first;

      removedFriends.push_back(kvp.second);

      auto mapIter = currentFriendMap.find(userId);
      TTV_ASSERT(mapIter != currentFriendMap.end());
      if (mapIter != currentFriendMap.end()) {
        currentFriendMap.erase(mapIter);
      }
    }
  } else {
    for (const auto& entry : list) {
      currentFriendMap[entry.friendData.userInfo.userId] = entry;
    }
  }

  // Persist the result we've built up
  mFriendMap = currentFriendMap;

  if (mReceivedFirstFriendListFetch) {
    // Notify of changes
    for (auto id : addedFriends) {
      mPendingInfoChanges.insert(id);

      AddToPendingFriendAdds(id);
    }

    for (const auto& entry : removedFriends) {
      AddToPendingFriendRemoves(entry);
    }
  } else {
    for (const auto& kvp : mFriendMap) {
      mPendingInfoChanges.insert(kvp.first);
    }
  }

  mReceivedFirstFriendListFetch = true;
}

std::vector<ttv::social::Friend> ttv::social::FriendList::BuildSocialFriendListResult(
  const std::unordered_set<UserId>& ids) {
  std::vector<Friend> list;
  for (auto userId : ids) {
    auto friendIter = mFriendMap.find(userId);
    if (friendIter != mFriendMap.end()) {
      list.push_back(friendIter->second.friendData);
    }
  }

  return list;
}

std::vector<ttv::social::Friend> ttv::social::FriendList::BuildSocialFriendListResult(
  const std::map<UserId, Friend>& users) {
  std::vector<Friend> result;
  std::transform(users.begin(), users.end(), std::back_inserter(result),
    [](const std::pair<UserId, Friend>& entry) { return entry.second; });

  return result;
}

void ttv::social::FriendList::NotifyFriendInfoChanges() {
  if (mPendingInfoChanges.empty()) {
    return;
  }

  auto result = BuildSocialFriendListResult(mPendingInfoChanges);

  mPendingInfoChanges.clear();

  // The data wasn't available for any of the users so don't fire the event
  if (result.size() == 0) {
    return;
  }

  // Notify the listeners
  mListeners.Invoke(
    [this, result](std::shared_ptr<IListener> listener) { listener->OnFriendInfoChanged(this, result); });
}

void ttv::social::FriendList::NotifyFriendAddsRemoves() {
  if (mPendingAddedFriends.empty() && mPendingRemovedFriends.empty()) {
    return;
  }

  auto addList = BuildSocialFriendListResult(mPendingAddedFriends);
  auto removeList = BuildSocialFriendListResult(mPendingRemovedFriends);

  mPendingAddedFriends.clear();
  mPendingRemovedFriends.clear();

  // Notify the listeners
  mListeners.Invoke([this, addList, removeList](std::shared_ptr<IListener> listener) {
    listener->OnFriendshipChanged(this, addList, removeList);
  });
}

TTV_ErrorCode ttv::social::FriendList::UpdateFriendship(
  UserId otherUserId, FriendAction action, const UpdateFriendshipCallback& callback) {
  if (!mEnabledFeatures.friendList) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  // If the user info hasn't been fetched yet we can't do anything
  else if (user->GetUserId() == 0) {
    return TTV_EC_USERINFO_NOT_AVAILABLE;
  }

  auto oauthToken = user->GetOAuthToken();
  SocialUpdateFriendTask::Action taskAction;
  switch (action) {
    case FriendAction::SendRequest:
      taskAction = SocialUpdateFriendTask::Action::SendRequest;
      break;
    case FriendAction::AcceptRequest:
      taskAction = SocialUpdateFriendTask::Action::AcceptRequest;
      break;
    case FriendAction::RejectRequest:
      taskAction = SocialUpdateFriendTask::Action::RejectRequest;
      break;
    case FriendAction::DeleteFriend:
      taskAction = SocialUpdateFriendTask::Action::DeleteFriend;
      break;
    default:
      return TTV_EC_INVALID_ARG;
  }

  std::shared_ptr<Task> task =
    std::make_shared<SocialUpdateFriendTask>(user->GetUserId(), oauthToken->GetToken(), otherUserId, taskAction,
      [this, taskAction, otherUserId, user, oauthToken, callback](SocialUpdateFriendTask* source, TTV_ErrorCode ec,
        const std::shared_ptr<SocialUpdateFriendTask::Result>& taskResult) {
        CompleteTask(source);

        UpdateFriendResult result = UpdateFriendResult::Unknown;
        FriendStatus status = FriendStatus::Unknown;

        if (TTV_SUCCEEDED(ec) && taskResult != nullptr) {
          result = taskResult->result;
          status = taskResult->status;

          switch (taskAction) {
            case SocialUpdateFriendTask::Action::AcceptRequest: {
              // The request was outstanding
              if (taskResult->result == UpdateFriendResult::RequestAccepted) {
                ec = HandleFriendRequestAccepted(otherUserId, FriendRequestRemovalReason::SelfAccepted);
              }
              // There was no request to accept
              else {
                // Remove from cached requests
                RemoveFriendRequest(otherUserId, FriendRequestRemovalReason::Invalid);
              }

              // Since we don't know if they're online or not we need to schedule a refresh of the friend list
              mFriendListFetchTimer.Set(kFriendAddedRefreshDelayMilliseconds);

              break;
            }
            case SocialUpdateFriendTask::Action::RejectRequest: {
              // NOTE: We don't use the result

              // Remove from the locally cached requests
              RemoveFriendRequest(otherUserId, FriendRequestRemovalReason::SelfRejected);

              break;
            }
            case SocialUpdateFriendTask::Action::SendRequest: {
              // NOTE: We don't use the result

              break;
            }
            case SocialUpdateFriendTask::Action::DeleteFriend: {
              // Remove from the friend list
              auto iter = mFriendMap.find(otherUserId);
              if (iter != mFriendMap.end()) {
                AddToPendingFriendRemoves(iter->second);

                mFriendMap.erase(iter);
              }

              break;
            }
            default: { break; }
          }
        } else if (ec == TTV_EC_AUTHENTICATION) {
          user->ReportOAuthTokenInvalid(oauthToken, ec);
        }

        if (callback != nullptr) {
          callback(ec, result, status);
        }
      });

  return StartTask(task);
}

TTV_ErrorCode ttv::social::FriendList::PerformFriendRequestListManagement(FriendRequestsTaskSetupCallback setupCallback,
  FriendRequestsTaskProcessResultCallback processCallback, CompleteCallback completeCallback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  auto complete = [cleanup, completeCallback](TTV_ErrorCode ec) {
    cleanup(ec);
    if (completeCallback != nullptr) {
      completeCallback(ec);
    }
  };

  SocialFriendRequestsTask::Callback resultCallback = [this, processCallback, cleanup, complete](
                                                        SocialFriendRequestsTask* source, TTV_ErrorCode ec,
                                                        std::shared_ptr<SocialFriendRequestsTask::Result> result) {
    CompleteTask(source);

    if (TTV_SUCCEEDED(ec) && result != nullptr) {
      ec = processCallback(result);
    } else {
      ec = TTV_EC_API_REQUEST_FAILED;
    }

    complete(ec);
  };

  auto task = std::make_shared<SocialFriendRequestsTask>(user->GetUserId(), oauthToken->GetToken(), resultCallback);
  setupCallback(task);

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    cleanup(ec);
  }

  return ec;
}

TTV_ErrorCode ttv::social::FriendList::FetchFriendRequests(const FetchFriendRequestsCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!mEnabledFeatures.friendRequests) {
    return TTV_EC_FEATURE_DISABLED;
  }

  mFetchFriendRequestsCallbacks.Push(callback);

  // Kick off a fetch immediately on an explicit request.
  mFriendRequestFetchTimer.Set(0);

  return TTV_EC_SUCCESS;
}

void ttv::social::FriendList::RequestFriendRequests() {
  if (mState != State::Initialized) {
    return;
  } else if (!mEnabledFeatures.friendRequests) {
    return;
  } else if (mFriendRequestFetcher->InProgress()) {
    return;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  struct Data {
    FriendRequestMap latestFriendRequests;
    bool done;
  };

  auto data = std::make_shared<Data>();
  data->done = false;

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [this, user, oauthToken](TTV_ErrorCode ec) {
    // Failed so try again soon
    if (TTV_FAILED(ec)) {
      mFriendRequestFetchTimer.SetWithJitter(kFailedRetryIntervalMilliseconds, kFailedRetryJitterMilliseconds);
    }
    // Do a full sync every so often
    else if (!mFriendListFetchTimer.IsSet()) {
      mFriendRequestFetchTimer.Set(kCoarseResyncIntervalMilliseconds);
    }

    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  PagedRequestFetcher::CompleteCallback completeCallback = [this, data, cleanup](TTV_ErrorCode ec) {
    if (TTV_SUCCEEDED(ec)) {
      mFriendRequests = data->latestFriendRequests;
      mReceivedFirstFriendRequestsFetch = true;
    }

    cleanup(ec);

    // Sanity check the unread friend request count
    if (TTV_SUCCEEDED(ec) && static_cast<uint32_t>(mFriendRequests.size()) < mNumUnreadFriendRequests) {
      SetNumUnreadFriendRequests(static_cast<uint32_t>(mFriendRequests.size()));
    }
  };

  SocialFriendRequestsTask::Callback resultCallback = [this, user, data, completeCallback](
                                                        SocialFriendRequestsTask* source, TTV_ErrorCode ec,
                                                        std::shared_ptr<SocialFriendRequestsTask::Result> result) {
    CompleteTask(source);

    data->done = true;
    std::string cursor = "";

    if (TTV_SUCCEEDED(ec)) {
      TTV_ASSERT(result != nullptr);

      for (const auto& request : result->requests) {
        FriendRequest entry;
        entry.userInfo = request.userInfo;
        entry.requestTime = request.requestedTimestamp;
        data->latestFriendRequests[request.userInfo.userId] = entry;
      }

      // Determine if there are more pages
      data->done = result->requests.size() < static_cast<size_t>(SocialFriendRequestsTask::kMaxLimit) ||
                   data->latestFriendRequests.size() >= static_cast<size_t>(result->total);

      if (!data->done) {
        cursor = result->cursor;
      }
    }

    // Notify the fetcher that we have the page result
    mFriendRequestFetcher->FetchComplete(ec, cursor);
  };

  PagedRequestFetcher::CreateTaskCallback createTaskCallback = [this, user, data, oauthToken, resultCallback](
                                                                 const std::string& cursor,
                                                                 std::shared_ptr<Task>& task) -> TTV_ErrorCode {
    task.reset();

    if (!data->done) {
      std::shared_ptr<SocialFriendRequestsTask> fetchTask =
        std::make_shared<SocialFriendRequestsTask>(user->GetUserId(), oauthToken->GetToken(), resultCallback);
      fetchTask->FetchRequests(
        SocialFriendRequestsTask::kMaxLimit, SocialFriendRequestsTask::SortDirection::Descending, cursor);

      TTV_ErrorCode ec = StartTask(fetchTask);
      if (TTV_SUCCEEDED(ec)) {
        task = fetchTask;
        return TTV_EC_SUCCESS;
      }
      return TTV_EC_SHUT_DOWN;
    } else {
      return TTV_EC_SUCCESS;
    }
  };

  mFriendRequestFetchTimer.Clear();

  TTV_ErrorCode ec = mFriendRequestFetcher->Start("", createTaskCallback, completeCallback);
  if (TTV_FAILED(ec)) {
    cleanup(ec);
  }
}

bool ttv::social::FriendList::RemoveFriendRequest(UserId userId, FriendRequestRemovalReason reason) {
  auto iter = mFriendRequests.find(userId);
  if (iter == mFriendRequests.end()) {
    return false;
  }

  UserInfo userInfo = iter->second.userInfo;

  mFriendRequests.erase(iter);

  // Notify clients
  mListeners.Invoke([this, reason, userInfo](std::shared_ptr<IListener> listener) {
    listener->OnFriendRequestRemoved(this, userInfo.userId, reason);
  });

  // Sanity check the unread friend request count
  if (static_cast<uint32_t>(mFriendRequests.size()) < mNumUnreadFriendRequests) {
    SetNumUnreadFriendRequests(static_cast<uint32_t>(mFriendRequests.size()));
  }

  return true;
}

TTV_ErrorCode ttv::social::FriendList::FetchUnreadFriendRequestCount(
  const FetchUnreadFriendRequestCountCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!mEnabledFeatures.friendRequests) {
    return TTV_EC_FEATURE_DISABLED;
  }

  mFetchUnreadFriendRequestCountCallbacks.Push(callback);

  // Kick off a fetch immediately on an explicit request.
  mFriendRequestUnreadCountFetchTimer.Set(0);

  return TTV_EC_SUCCESS;
}

void ttv::social::FriendList::RequestFriendRequestUnreadCount() {
  if (!mEnabledFeatures.friendRequests) {
    return;
  }

  FriendRequestsTaskProcessResultCallback processCallback =
    [this](std::shared_ptr<SocialFriendRequestsTask::Result> result) -> TTV_ErrorCode {
    SetNumUnreadFriendRequests(static_cast<uint32_t>(result->total));
    mReceivedFirstFriendRequestUnreadCountFetch = true;

    return TTV_EC_SUCCESS;
  };

  FriendRequestsTaskSetupCallback setupCallback = [](std::shared_ptr<SocialFriendRequestsTask> task) {
    task->GetUnreadCount();
  };

  CompleteCallback completeCallback = [this](TTV_ErrorCode ec) {
    if (TTV_SUCCEEDED(ec)) {
      mFriendRequestUnreadCountFetchTimer.Set(kCoarseResyncIntervalMilliseconds);
    } else {
      mFriendRequestUnreadCountFetchTimer.SetWithJitter(
        kFailedRetryIntervalMilliseconds, kFailedRetryJitterMilliseconds);
    }
  };

  PerformFriendRequestListManagement(setupCallback, processCallback, nullptr);
}

TTV_ErrorCode ttv::social::FriendList::MarkAllFriendRequestsRead(const MarkAllFriendRequestsReadCallback& callback) {
  if (!mEnabledFeatures.friendRequests) {
    return TTV_EC_FEATURE_DISABLED;
  }

  CompleteCallback completeCallback = [callback](TTV_ErrorCode ec) {
    if (callback != nullptr) {
      callback(ec);
    }
  };

  FriendRequestsTaskProcessResultCallback processCallback =
    [this](std::shared_ptr<SocialFriendRequestsTask::Result> /*result*/) -> TTV_ErrorCode {
    SetNumUnreadFriendRequests(0);

    return TTV_EC_SUCCESS;
  };

  FriendRequestsTaskSetupCallback setupCallback = [](std::shared_ptr<SocialFriendRequestsTask> task) {
    task->MarkAllRead();
  };

  return PerformFriendRequestListManagement(setupCallback, processCallback, completeCallback);
}

TTV_ErrorCode ttv::social::FriendList::FetchFriendStatus(
  UserId otherUserId, const FetchFriendStatusCallback& callback) {
  if (!mEnabledFeatures.friendList) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  // If the user info hasn't been fetched yet we can't do anything
  else if (user->GetUserId() == 0) {
    return TTV_EC_USERINFO_NOT_AVAILABLE;
  }

  auto oauthToken = user->GetOAuthToken();
  std::shared_ptr<Task> task = std::make_shared<SocialUpdateFriendTask>(user->GetUserId(), oauthToken->GetToken(),
    otherUserId, SocialUpdateFriendTask::Action::GetStatus,
    [this, user, oauthToken, callback](SocialUpdateFriendTask* source, TTV_ErrorCode ec,
      const std::shared_ptr<SocialUpdateFriendTask::Result>& taskResult) {
      CompleteTask(source);

      FriendStatus status = FriendStatus::Unknown;
      if (TTV_SUCCEEDED(ec) && (taskResult != nullptr)) {
        status = taskResult->status;
      } else if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      if (callback != nullptr) {
        callback(ec, status);
      }
    });

  return StartTask(task);
}

TTV_ErrorCode ttv::social::FriendList::FetchRecommendedFriends(const FetchRecommendedFriendsCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  mFetchRecommendedFriendsCallbacks.Push(callback);

  // Kick off a fetch immediately on an explicit request.
  mRecommendedFriendsFetchTimer.Set(0);

  return TTV_EC_SUCCESS;
}

void ttv::social::FriendList::RequestRecommendedFriends() {
  auto user = mUser.lock();
  if (user == nullptr) {
    return;
  }

  auto oauthToken = user->GetOAuthToken();

  auto complete = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  SocialRecommendedFriendsTask::Callback resultCallback =
    [this, complete](SocialRecommendedFriendsTask* source, TTV_ErrorCode ec,
      std::shared_ptr<SocialRecommendedFriendsTask::Result> result) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec)) {
        TTV_ASSERT(result != nullptr);
        TTV_ASSERT(result->action == SocialRecommendedFriendsTask::Action::Fetch);

        mRecommendedFriends.clear();

        for (const auto& rf : result->recommendedFriends) {
          mRecommendedFriends.push_back(rf.userInfo);
        }

        mReceivedFirstRecommendedFriendsFetch = true;
        mRecommendedFriendsFetchTimer.Set(kCoarseResyncIntervalMilliseconds);
      } else {
        mRecommendedFriendsFetchTimer.SetWithJitter(kFailedRetryIntervalMilliseconds, kFailedRetryJitterMilliseconds);
      }

      complete(ec);
    };

  auto task = std::make_shared<SocialRecommendedFriendsTask>(user->GetUserId(), oauthToken->GetToken(), resultCallback);
  task->Fetch();

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    complete(ec);
  }
}

TTV_ErrorCode ttv::social::FriendList::DismissRecommendedFriend(
  UserId dismissUserId, const DismissRecommendedFriendCallback& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto complete = [user, oauthToken, callback](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }

    if (callback != nullptr) {
      callback(ec);
    }
  };

  SocialRecommendedFriendsTask::Callback resultCallback =
    [this, complete, dismissUserId](SocialRecommendedFriendsTask* source, TTV_ErrorCode ec,
      std::shared_ptr<SocialRecommendedFriendsTask::Result> result) {
      CompleteTask(source);

      if (TTV_SUCCEEDED(ec)) {
        TTV_ASSERT(result != nullptr);
        TTV_ASSERT(result->action == SocialRecommendedFriendsTask::Action::Dismiss);

        std::remove_if(mRecommendedFriends.begin(), mRecommendedFriends.end(),
          [dismissUserId](const UserInfo& entry) { return entry.userId == dismissUserId; });
      }

      complete(ec);
    };

  auto task = std::make_shared<SocialRecommendedFriendsTask>(user->GetUserId(), oauthToken->GetToken(), resultCallback);
  task->Dismiss(dismissUserId);

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    complete(ec);
  }

  return ec;
}

void ttv::social::FriendList::SetTaskRunner(std::shared_ptr<TaskRunner> taskRunner) {
  UserComponent::SetTaskRunner(taskRunner);
}

void ttv::social::FriendList::Update() {
  switch (mState) {
    case State::Initialized: {
      if (!mOAuthIssue) {
        // Check if it's time to fetch the friend list
        if (mFriendListFetchTimer.Check(true)) {
          RequestFriendsList();
        }

        // Check if it's time to grab the latest list of friend requests
        if (mFriendRequestFetchTimer.Check(true)) {
          RequestFriendRequests();
        }

        if (mFriendRequestUnreadCountFetchTimer.Check(true)) {
          RequestFriendRequestUnreadCount();
        }

        if (mRecommendedFriendsFetchTimer.Check(true)) {
          RequestRecommendedFriends();
        }
      }

      if (mReceivedFirstFriendListFetch && !mFetchFriendListCallbacks.Empty()) {
        std::vector<Friend> friends;
        std::transform(mFriendMap.begin(), mFriendMap.end(), std::back_inserter(friends),
          [](const std::pair<UserId, FriendEntry>& pair) { return pair.second.friendData; });
        mFetchFriendListCallbacks.Flush(TTV_EC_SUCCESS, friends);
      }

      if (mReceivedFirstFriendRequestsFetch && !mFetchFriendRequestsCallbacks.Empty()) {
        std::vector<FriendRequest> requests;
        std::transform(mFriendRequests.begin(), mFriendRequests.end(), std::back_inserter(requests),
          [](const std::pair<UserId, FriendRequest>& pair) { return pair.second; });
        mFetchFriendRequestsCallbacks.Flush(TTV_EC_SUCCESS, requests);
      }

      if (mReceivedFirstFriendRequestUnreadCountFetch && !mFetchUnreadFriendRequestCountCallbacks.Empty()) {
        mFetchUnreadFriendRequestCountCallbacks.Flush(TTV_EC_SUCCESS, mNumUnreadFriendRequests);
      }

      if (mReceivedFirstRecommendedFriendsFetch && !mFetchRecommendedFriendsCallbacks.Empty()) {
        mFetchRecommendedFriendsCallbacks.Flush(TTV_EC_SUCCESS, mRecommendedFriends);
      }

      // Fire events for changes
      NotifyFriendAddsRemoves();
      NotifyFriendInfoChanges();

      break;
    }
    default: { break; }
  }

  Component::Update();
}

TTV_ErrorCode ttv::social::FriendList::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mPubSubTopicListenerHelper != nullptr) {
      mPubSubTopicListenerHelper->Shutdown();
    }

    mFriendRequestFetcher->Cancel();
  }

  return ec;
}

bool ttv::social::FriendList::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  // Make sure unsubscribed from pubsub topics
  if (mPubSubTopicListenerHelper != nullptr &&
      mPubSubTopicListenerHelper->GetState() != PubSubTopicListenerHelper::State::Shutdown) {
    return false;
  }

  return true;
}

std::string ttv::social::FriendList::GetLoggerName() const {
  return kLogger;
}

void ttv::social::FriendList::CompleteShutdown() {
  UserComponent::CompleteShutdown();

  TTV_ASSERT(!mFriendRequestFetcher->InProgress());

  mUserRepository.reset();
  mPubSub.reset();
  mPubSubTopicListener.reset();
  mPubSubTopicListenerHelper.reset();
  mListeners.ClearListeners();
}

void ttv::social::FriendList::OnUserInfoFetchComplete(TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec)) {
    SubscribeTopics();
  }

  UserComponent::OnUserInfoFetchComplete(ec);
}

TTV_ErrorCode ttv::social::FriendList::HandleFriendRequestAccepted(UserId userId, FriendRequestRemovalReason reason) {
  std::vector<UserId> userIds;
  userIds.push_back(userId);

  auto completion = [this, reason](const UserInfo& userInfo) {
    auto iter = mFriendMap.find(userInfo.userId);
    if (iter == mFriendMap.end()) {
      auto entry = std::make_shared<FriendEntry>();
      entry->friendData.userInfo = userInfo;
      entry->friendData.friendsSinceTime = GetCurrentTimeAsUnixTimestamp();

      // Add the friend to the friend list
      mFriendMap[userInfo.userId] = *entry;

      RemoveFriendRequest(userInfo.userId, reason);
      AddToPendingFriendAdds(userInfo.userId);
    } else {
      iter->second.friendData.userInfo = userInfo;
    }
  };

  // Since we don't know if they're online or not we need to schedule a refresh of the friend list
  mFriendListFetchTimer.Set(kFriendAddedRefreshDelayMilliseconds);

  UserInfo userInfo;
  if (TTV_SUCCEEDED(mUserRepository->GetUserInfoById(userId, userInfo))) {
    completion(userInfo);
    return TTV_EC_SUCCESS;
  } else {
    return mUserRepository->FetchUserInfoById(
      userId, [completion](const ErrorDetails& /*errorDetails*/, const UserInfo& result) { completion(result); });
  }
}

void ttv::social::FriendList::HandleFriendRemoved(UserId userId) {
  auto iter = mFriendMap.find(userId);
  if (iter != mFriendMap.end()) {
    AddToPendingFriendRemoves(iter->second);

    mFriendMap.erase(iter);
  }
}

TTV_ErrorCode ttv::social::FriendList::HandleFriendRequestRemoved(UserId userId, FriendRequestRemovalReason reason) {
  // Kick off a simple defer task in order to do processing in the main thread
  LambdaTask::CompleteCallback callback = [this, userId, reason](LambdaTask* source, TTV_ErrorCode ec) {
    CompleteTask(source);

    if (TTV_SUCCEEDED(ec)) {
      RemoveFriendRequest(userId, reason);
    }
  };

  std::shared_ptr<Task> task = std::make_shared<LambdaTask>(nullptr, callback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start LambdaTask");
  }

  return ec;
}

TTV_ErrorCode ttv::social::FriendList::HandleRealtimeFriendRequestReceived(UserId userId) {
  if (!mEnabledFeatures.friendRequests) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto getUserInfoCompleteCallback = [this](const UserInfo& userInfo) {
    FriendRequest request;
    request.requestTime = GetCurrentTimeAsUnixTimestamp();
    request.userInfo = userInfo;

    mFriendRequests[userInfo.userId] = request;

    mListeners.Invoke([this, &request](std::shared_ptr<IListener> listener) {
      listener->OnRealtimeFriendRequestReceived(this, request);
    });

    SetNumUnreadFriendRequests(mNumUnreadFriendRequests + 1);
  };

  UserInfo info;
  if (TTV_SUCCEEDED(mUserRepository->GetUserInfoById(userId, info))) {
    getUserInfoCompleteCallback(info);
    return TTV_EC_SUCCESS;
  } else {
    return mUserRepository->FetchUserInfoById(
      userId, [getUserInfoCompleteCallback](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
        if (TTV_SUCCEEDED(errorDetails.ec)) {
          getUserInfoCompleteCallback(userInfo);
        }
      });
  }
}

void ttv::social::FriendList::SetNumUnreadFriendRequests(uint32_t count) {
  if (mNumUnreadFriendRequests == count) {
    return;
  }

  mNumUnreadFriendRequests = count;

  mListeners.Invoke(
    [this, count](std::shared_ptr<IListener> listener) { listener->OnUnreadFriendRequestCountChanged(this, count); });
}

void ttv::social::FriendList::OnTopicSubscribeStateChanged(
  const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  if (state == PubSubClient::SubscribeState::Subscribed) {
    // Resync all data since we may have missed incremental updates
    // NOTE: Only do it for the friendship topic since we expect the presence topic to fire the same event
    // and FetchFriendList() will refresh presence as well.
    if (topic == mFriendshipTopic) {
      RequestFriendsList();
      RequestFriendRequests();
    }
  }
}

void ttv::social::FriendList::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPresenceTopic) {
    const auto& jType = msg["type"];
    if (jType.isNull() || !jType.isString()) {
      Log(MessageLevel::Info, "No 'type' field, ignoring");
      return;
    }

    // We only care about friend presence updates
    if (jType.asString() != "presence") {
      return;
    }

    const auto& jData = msg["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Info, "No 'data' field, ignoring");
      return;
    }

    const auto& jUserId = jData["user_id"];
    const auto& jIndex = jData["index"];
    const auto& jUpdated = jData["updated_at"];
    const auto& jAvailability = jData["availability"];

    if (jUserId.isNull() || !jUserId.isNumeric() || jIndex.isNull() || !jIndex.isNumeric() || jUpdated.isNull() ||
        !jUpdated.isNumeric() || jAvailability.isNull() || !jAvailability.isString()) {
      Log(MessageLevel::Error, "Invalid presence json from pubsub");
      return;
    }

    UserId userId = static_cast<UserId>(jUserId.asUInt());
    if (userId == 0) {
      Log(MessageLevel::Error, "Invalid user id");
      return;
    }

    PresenceUserAvailability availability;
    if (!ParsePresenceUserAvailability(jAvailability.asString(), availability)) {
      Log(MessageLevel::Error, "Unhandled 'availability': %s", jAvailability.asString().c_str());
    }

    std::unique_ptr<PresenceActivity> activity;
    const auto& jActivity = jData["activity"];
    if (!jActivity.isNull() && jActivity.isObject()) {
      CreatePresenceActivity(jActivity, activity);
    }

    uint64_t updateIndex = static_cast<uint64_t>(jIndex.asUInt());

    Timestamp lastUpdateTime = static_cast<Timestamp>(jUpdated.asUInt());
    SetFriendPresence(userId, lastUpdateTime, updateIndex, availability, std::move(activity));
  } else if (topic == mFriendshipTopic) {
    const auto& jUserId = msg["user_id"];
    const auto& jChange = msg["change"];

    UserId userId = 0;
    if (!ParseUserId(jUserId, userId)) {
      Log(MessageLevel::Error, "Invalid user_id");
    }

    if (jChange.isNull() || !jChange.isString()) {
      Log(MessageLevel::Error, "Invalid frienship json from pubsub");
      return;
    }

    std::string change = jChange.asString();

    // Received a friend request from another user
    if (change == "requested") {
      HandleRealtimeFriendRequestReceived(userId);
    }
    // The other user has accepted our request
    else if (change == "accepted") {
      HandleFriendRequestAccepted(userId, FriendRequestRemovalReason::TargetAccepted);
    }
    // You have accepted a friend request, possibly from another device
    else if (change == "self_accepted") {
      UserId targetUserId = 0;
      const auto& jTargetUserId = msg["target_user_id"];

      if (!ParseUserId(jTargetUserId, targetUserId)) {
        Log(MessageLevel::Error, "Invalid target_user_id");
      }

      HandleFriendRequestAccepted(targetUserId, FriendRequestRemovalReason::SelfAccepted);
    }
    // The other user has rejected your friend request
    else if (change == "rejected") {
      HandleFriendRequestRemoved(userId, FriendRequestRemovalReason::TargetRejected);
    }
    // You have rejected a friend request, possibly from another device
    else if (change == "self_rejected") {
      UserId targetUserId = 0;
      const auto& jTargetUserId = msg["target_user_id"];

      if (!ParseUserId(jTargetUserId, targetUserId)) {
        Log(MessageLevel::Error, "Invalid target_user_id");
      }

      HandleFriendRequestRemoved(targetUserId, FriendRequestRemovalReason::SelfRejected);
    }
    // Another user has removed you as a friend
    else if (change == "removed") {
      HandleFriendRemoved(userId);
    }
    // You have removed a friend, possibly from another device
    else if (change == "self_removed") {
      UserId targetUserId = 0;
      const auto& jTargetUserId = msg["target_user_id"];
      if (!ParseUserId(jTargetUserId, targetUserId)) {
        Log(MessageLevel::Error, "Invalid target_user_id");
      }

      HandleFriendRemoved(targetUserId);
    } else {
      Log(MessageLevel::Info, "Unhandled %s change received via pubsub: %s", mFriendshipTopic.c_str(), change.c_str());
    }
  }
}

ttv::social::FriendList::PubSubTopicListener::PubSubTopicListener(FriendList* owner) : mOwner(owner) {}

void ttv::social::FriendList::PubSubTopicListener::OnTopicSubscribeStateChanged(
  PubSubClient* /*source*/, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  mOwner->OnTopicSubscribeStateChanged(topic, state, ec);
}

void ttv::social::FriendList::PubSubTopicListener::OnTopicMessageReceived(
  PubSubClient* /*source*/, const std::string& topic, const json::Value& msg) {
  mOwner->OnTopicMessageReceived(topic, msg);
}

void ttv::social::FriendList::PubSubTopicListener::OnTopicListenerRemoved(
  PubSubClient* /*source*/, const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {
  // NOTE: The listener is only removed if we requested it to be removed or pubsub is shutting down
}
