/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/presence.h"

#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/random.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/graphqltask.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/social/internal/graphql/generated/setsessionstatusqueryinfo.h"
#include "twitchsdk/social/internal/socialhelpers.h"
#include "twitchsdk/social/internal/task/socialpresencesettingstask.h"

#include <ctime>

namespace {
const char* kLogger = "Presence";

uint64_t kDefaultPostIntervalMilliseconds = 60 * 1000;
uint64_t kDefaultPostRetryMilliseconds = 30 * 1000;
uint64_t kDefaultSettingsFetchRetryMilliseconds = 10 * 1000;
uint64_t kPostPresenceDelayMilliseconds = 5 * 1000;
}  // namespace

ttv::social::Presence::Presence(std::shared_ptr<User> user)
    : UserComponent(user),
      mPostIntervalMilliseconds(kDefaultPostIntervalMilliseconds),
      mActivityTokenCounter(1),
      mAvailability(PresenceSessionAvailability::Online),
      mEnableAutomaticPosting(true) {
  // Create a new GUID for the session
  char buffer[33];
  (void)snprintf(buffer, sizeof(buffer), "%llu", static_cast<unsigned long long>(GetSystemClockTime()));
  buffer[sizeof(buffer) - 1] = 0;

  size_t len = strlen(buffer);

  std::uniform_int_distribution<int> distribution(0, 9);
  for (size_t i = len; i < sizeof(buffer) - 1; ++i) {
    buffer[i] = static_cast<char>('0' + distribution(ttv::random::GetGenerator()));
  }

  mSessionGuid = buffer;

  mPubSub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (mPubSub != nullptr) {
    mPubSubTopicListener = std::make_shared<PubSubTopicListener>(this);
    mPubSubTopicListenerHelper = std::make_shared<PubSubTopicListenerHelper>(mPubSub, mPubSubTopicListener);
  }
}

ttv::social::Presence::~Presence() {
  Log(MessageLevel::Debug, "Presence - dtor");
}

void ttv::social::Presence::AddListener(std::shared_ptr<IListener> listener) {
  mListeners.AddListener(listener);
}

void ttv::social::Presence::RemoveListener(std::shared_ptr<IListener> listener) {
  mListeners.RemoveListener(listener);
}

TTV_ErrorCode ttv::social::Presence::Initialize() {
  TTV_ErrorCode ec = UserComponent::Initialize();
  if (TTV_FAILED(ec)) {
    return ec;
  }

  // Post our presence
  mPresencePostTimer.Set(0);

  // Kick off an immediate fetch of the current settings
  mSettingsFetchTimer.Set(0);

  // Attempt to subscribe to needed topics
  SubscribeTopics();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::SubscribeTopics() {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (mPubSubTopicListenerHelper == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  } else if (user->GetUserId() == 0) {
    Log(MessageLevel::Debug, "UserId not yet retrieved");

    return TTV_EC_USERINFO_NOT_AVAILABLE;
  }

  if (mPresenceTopic == "") {
    // Generate the topic ids we need to subscribe to
    char buffer[64];

    snprintf(buffer, sizeof(buffer), "presence.%u", user->GetUserId());
    mPresenceTopic = buffer;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  if (!mPubSubTopicListenerHelper->ContainsTopic(mPresenceTopic)) {
    ec = mPubSubTopicListenerHelper->Subscribe(mPresenceTopic);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  return ec;
}

void ttv::social::Presence::Update() {
  switch (mState.client) {
    case State::Initialized: {
      if (!mOAuthIssue) {
        if (mSettingsFetchTimer.Check(true)) {
          TTV_ErrorCode ec = UpdateSettings();
          if (TTV_FAILED(ec)) {
            mSettingsFetchTimer.Set(kDefaultSettingsFetchRetryMilliseconds);
          }
        }

        // Check to see if it's time to repost our presence
        if (mEnableAutomaticPosting && mPresencePostTimer.Check(true)) {
          PostPresence(nullptr);
        }
      }

      if (mSettings != nullptr) {
        mFetchSettingsCallbacks.Flush(TTV_EC_SUCCESS, *mSettings);
      }

      break;
    }
    default: { break; }
  }

  Component::Update();
}

TTV_ErrorCode ttv::social::Presence::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mPubSubTopicListenerHelper != nullptr) {
      mPubSubTopicListenerHelper->Shutdown();
    }
  }

  return ec;
}

bool ttv::social::Presence::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  // Make sure unsubscribed from pubsub topics
  if (mPubSubTopicListenerHelper != nullptr &&
      mPubSubTopicListenerHelper->GetState() != PubSubTopicListenerHelper::State::Shutdown) {
    return false;
  }

  return true;
}

void ttv::social::Presence::CompleteShutdown() {
  UserComponent::CompleteShutdown();

  TTV_ASSERT(mSocialSettingsTask == nullptr);
  TTV_ASSERT(mSocialPostPresenceTask == nullptr);

  mPubSub.reset();
  mUserRepository.reset();
  mPubSubTopicListener.reset();
  mPubSubTopicListenerHelper.reset();
  mListeners.ClearListeners();
}

std::string ttv::social::Presence::GetLoggerName() const {
  return kLogger;
}

TTV_ErrorCode ttv::social::Presence::PostPresence(const PostPresenceCallback& callback) {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  } else if (mSocialPostPresenceTask != nullptr) {
    return TTV_EC_REQUEST_PENDING;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();
  auto availability = mAvailability;

  auto completeCallback = [this, user, oauthToken, availability, callback](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }

    // We need to post the availability change if it happened while the previous post was in progress
    if (TTV_SUCCEEDED(ec) && mAvailability != availability) {
      mPresencePostTimer.Set(0);
    } else if (TTV_FAILED(ec)) {
      mPresencePostTimer.Set(kDefaultPostRetryMilliseconds);
    }

    if (callback != nullptr) {
      callback(ec);
    }
  };

  std::vector<std::unique_ptr<PresenceActivity>> activities;
  std::transform(mActivities.begin(), mActivities.end(), std::back_inserter(activities),
    [](const std::pair<const uint32_t, std::unique_ptr<PresenceActivity>>& pair) {
      TTV_ASSERT(pair.second != nullptr);
      return pair.second->Clone();
    });

  ttv::trace::Message(kLogger, MessageLevel::Debug, "Presence::PostPresence");

  graphql::SetSessionStatusQueryInfo::InputParams inputParams;
  inputParams.authToken = oauthToken->GetToken();

  inputParams.sessionId = mSessionGuid;

  // Get the availabilitity
  inputParams.availability = [this, &availability]() -> graphql::SetSessionStatusQueryInfo::AvailabilityInput {
    switch (availability) {
      case PresenceSessionAvailability::Idle:
        return graphql::SetSessionStatusQueryInfo::AvailabilityInput::IDLE;
      case PresenceSessionAvailability::Offline:
        return graphql::SetSessionStatusQueryInfo::AvailabilityInput::OFFLINE;
      case PresenceSessionAvailability::Online:
        return graphql::SetSessionStatusQueryInfo::AvailabilityInput::ONLINE;
    }
    // no default to allow compiler to detect missing enums
    Log(ttv::MessageLevel::Warning, "Unknown Availability %d", static_cast<int>(availability));
    return graphql::SetSessionStatusQueryInfo::AvailabilityInput::ONLINE;
  }();

  // Get the most recent valid activity
  // Activities are sort with more recently added last
  [&activities, &inputParams]() {
    for (auto itr = activities.rbegin(); itr != activities.rend(); itr++) {
      const auto* activity = itr->get();
      switch (activity->GetType()) {
        case PresenceActivity::Type::Broadcasting: {
          const BroadcastingActivity* broadcastActivity = static_cast<const BroadcastingActivity*>(activity);
          inputParams.activityType = graphql::SetSessionStatusQueryInfo::ActivityType::STREAMING;
          inputParams.activityGameID = std::to_string(broadcastActivity->gameId);
          return;
        }
        case PresenceActivity::Type::Playing: {
          const PlayingActivity* playingActivity = static_cast<const PlayingActivity*>(activity);
          inputParams.activityType = graphql::SetSessionStatusQueryInfo::ActivityType::PLAYING;
          inputParams.activityGameID = std::to_string(playingActivity->gameId);
          return;
        }
        case PresenceActivity::Type::Watching: {
          const WatchingActivity* watchingActivity = static_cast<const WatchingActivity*>(activity);
          inputParams.activityType = graphql::SetSessionStatusQueryInfo::ActivityType::WATCHING;
          inputParams.activityUserId = std::to_string(watchingActivity->channelId);
          return;
        }
        case PresenceActivity::Type::Unknown:
          break;
      }
    }
  }();

  auto resultCallback = [this, completeCallback](GraphQLTask<graphql::SetSessionStatusQueryInfo>* source,
                          Result<graphql::SetSessionStatusQueryInfo::PayloadType>&& result) {
    CompleteTask(source);

    mSocialPostPresenceTask.reset();

    if (result.IsError()) {
      Log(MessageLevel::Debug, "SetSessionStatusQueryInfo failed");
      completeCallback(result.GetErrorCode());
      return;
    }

    const auto& data = result.GetResult();
    mPostIntervalMilliseconds = static_cast<uint64_t>(data.setAgainInSeconds) * 1000;
    mPresencePostTimer.Set(mPostIntervalMilliseconds);

    completeCallback(result.GetErrorCode());
  };

  mSocialPostPresenceTask =
    std::make_shared<GraphQLTask<graphql::SetSessionStatusQueryInfo>>(std::move(inputParams), resultCallback);

  TTV_ErrorCode ec = StartTask(mSocialPostPresenceTask);

  if (TTV_SUCCEEDED(ec)) {
    mPresencePostTimer.Clear();
  } else {
    mSocialPostPresenceTask.reset();
    mPresencePostTimer.Set(kDefaultPostRetryMilliseconds);

    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  }

  return ec;
}

TTV_ErrorCode ttv::social::Presence::SetSessionAvailability(PresenceSessionAvailability availability) {
  if (availability != mAvailability) {
    mAvailability = availability;

    PresenceChanged();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::AddWatchingActivity(ChannelId channelId, PresenceActivityToken& activityToken) {
  if (channelId == 0) {
    return TTV_EC_INVALID_ARG;
  }

  auto activity = std::make_unique<WatchingActivity>();
  activity->channelId = channelId;

  activityToken = InsertActivity(std::move(activity));

  PresenceChanged();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::AddPlayingActivity(
  GameId gameId, const std::string& context, PresenceActivityToken& activityToken) {
  if (gameId == 0) {
    return TTV_EC_INVALID_ARG;
  }

  auto activity = std::make_unique<PlayingActivity>();
  activity->gameId = gameId;
  activity->gameDisplayContext = context;

  activityToken = InsertActivity(std::move(activity));

  PresenceChanged();

  return TTV_EC_SUCCESS;
}

uint32_t ttv::social::Presence::InsertActivity(std::unique_ptr<PresenceActivity>&& activity) {
  auto activityToken = mActivityTokenCounter;
  mActivityTokenCounter++;
  mActivities[activityToken] = std::move(activity);
  return activityToken;
}

TTV_ErrorCode ttv::social::Presence::RemoveActivity(PresenceActivityToken activityToken) {
  auto activityIterator = mActivities.find(activityToken);
  if (activityIterator == mActivities.end()) {
    return TTV_EC_INVALID_ARG;
  }

  mActivities.erase(activityIterator);

  PresenceChanged();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::SetPresenceSettings(
  const PresenceSettings& settings, const SetPresenceSettingsCallback& callback) {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  } else if (mSocialSettingsTask != nullptr) {
    return TTV_EC_REQUEST_PENDING;
  }

  auto user = mUser.lock();
  auto oauthToken = user->GetOAuthToken();
  if ((user == nullptr) || (!oauthToken->GetValid())) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto task = std::make_shared<SocialPresenceSettingsTask>(user->GetUserId(), oauthToken->GetToken(), true,
    [this, callback](SocialPresenceSettingsTask* source, TTV_ErrorCode ec,
      const std::shared_ptr<SocialPresenceSettingsTask::Result>& /*result*/) {
      CompleteTask(source);

      if (callback != nullptr) {
        callback(ec);
      }

      mSocialSettingsTask.reset();
    });
  task->SetSettings(settings);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_SUCCEEDED(ec)) {
    mSocialSettingsTask = task;
  }
  return ec;
}

TTV_ErrorCode ttv::social::Presence::SetAutomaticPresencePostingEnabled(bool enabled) {
  if (enabled != mEnableAutomaticPosting) {
    mEnableAutomaticPosting = enabled;

    // Start the presence timer if needed
    if (enabled && !mPresencePostTimer.IsSet()) {
      mPresencePostTimer.Set(mPostIntervalMilliseconds);
    }
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::GetAutomaticPresencePostingEnabled(bool& enabled) {
  enabled = mEnableAutomaticPosting;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::FetchSettings(const FetchPresenceSettingsCallback& callback) {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  mFetchSettingsCallbacks.Push(callback);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::Presence::UpdateSettings() {
  if (mState.client != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  } else if (mSocialSettingsTask != nullptr) {
    return TTV_EC_REQUEST_PENDING;
  }

  auto user = mUser.lock();
  auto oauthToken = user->GetOAuthToken();
  if ((user == nullptr) || (!oauthToken->GetValid())) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto task = std::make_shared<SocialPresenceSettingsTask>(user->GetUserId(), oauthToken->GetToken(), false,
    [this, user, oauthToken](SocialPresenceSettingsTask* source, TTV_ErrorCode ec,
      const std::shared_ptr<SocialPresenceSettingsTask::Result>& result) {
      CompleteTask(source);
      if (TTV_SUCCEEDED(ec)) {
        SetSettings(result->settings);
      } else {
        mSettingsFetchTimer.Set(kDefaultSettingsFetchRetryMilliseconds);
      }

      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }
    });

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    mSettingsFetchTimer.Set(kDefaultSettingsFetchRetryMilliseconds);
  }

  return ec;
}

void ttv::social::Presence::SetSettings(const PresenceSettings& settings) {
  if ((mSettings != nullptr) && (*mSettings == settings)) {
    return;
  }

  mSettings = std::make_unique<PresenceSettings>(settings);

  // Notify listeners
  mListeners.Invoke(
    [this, &settings](std::shared_ptr<IListener> listener) { listener->OnSettingsChanged(this, settings); });
}

void ttv::social::Presence::PresenceChanged() {
  if (mEnableAutomaticPosting) {
    // If we don't already have a presence post scheduled in the next interval,
    // schedule it.
    if (mPresencePostTimer.GetRemainingTime() < kPostPresenceDelayMilliseconds) {
      mPresencePostTimer.Set(kPostPresenceDelayMilliseconds);
    }
  }
}

void ttv::social::Presence::OnTopicSubscribeStateChanged(
  const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec) && (mState.client != State::Initialized) && (topic == mPresenceTopic) &&
      (state == PubSubClient::SubscribeState::Enum::Subscribed)) {
    // If we just subscribed successfully, we need to make sure to fetch our settings in case we missed anything.
    UpdateSettings();
  }
}

void ttv::social::Presence::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPresenceTopic) {
    const auto& jType = msg["type"];
    if (jType.isNull() || !jType.isString()) {
      Log(MessageLevel::Info, "No 'type' field, ignoring");
      return;
    }

    // We only care about settings
    if (jType.asString() != "settings") {
      return;
    }

    const auto& jData = msg["data"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Info, "No 'data' field, ignoring");
      return;
    }

    const auto& jAvailabilityOverride = jData["availability_override"];
    const auto& jShareActivity = jData["share_activity"];
    if (jAvailabilityOverride.isNull() || !jAvailabilityOverride.isString() || jShareActivity.isNull() ||
        !jShareActivity.isBool()) {
      Log(MessageLevel::Error, "Invalid presence json from pubsub");
      return;
    }

    PresenceSettings settings;
    ParsePresenceSettingsAvailabilityOverride(jAvailabilityOverride.asString(), settings.availabilityOverride);
    settings.shareActivity = jShareActivity.asBool();
    SetSettings(settings);
  }
}

void ttv::social::Presence::OnTopicListenerRemoved(const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {
  // NOTE: The listener is only removed if we requested it to be removed or pubsub is shutting down
}

void ttv::social::Presence::OnUserInfoFetchComplete(TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec)) {
    SubscribeTopics();
  }

  UserComponent::OnUserInfoFetchComplete(ec);
}

ttv::social::Presence::PubSubTopicListener::PubSubTopicListener(Presence* owner) : mOwner(owner) {}

void ttv::social::Presence::PubSubTopicListener::OnTopicSubscribeStateChanged(
  PubSubClient* /*source*/, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  mOwner->OnTopicSubscribeStateChanged(topic, state, ec);
}

void ttv::social::Presence::PubSubTopicListener::OnTopicMessageReceived(
  PubSubClient* /*source*/, const std::string& topic, const json::Value& msg) {
  mOwner->OnTopicMessageReceived(topic, msg);
}

void ttv::social::Presence::PubSubTopicListener::OnTopicListenerRemoved(
  PubSubClient* /*source*/, const std::string& topic, TTV_ErrorCode ec) {
  mOwner->OnTopicListenerRemoved(topic, ec);
}
