/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/socialapi.h"

#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/types/memorytypes.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/social/internal/friendlist.h"
#include "twitchsdk/social/internal/presence.h"
#include "twitchsdk/social/sociallistener.h"

namespace {
const char* kModule = "ttv::social::SocialAPI";
const char* kRequiredScopes[] = {
  "user_read", "user_friends_read", "user_friends_edit", "user_presence_friends_read", "user_presence_edit"};
}  // namespace

struct ttv::social::SocialAPIInternalData {
  std::shared_ptr<FriendListListenerProxy> mFriendListListener;
  std::shared_ptr<PresenceListenerProxy> mPresenceListener;
};

std::string ttv::social::SocialAPI::CoreApiClient::GetClientName() {
  return kModule;
}

void ttv::social::SocialAPI::CoreApiClient::GetRequiredOAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

ttv::social::SocialAPI::SocialAPI() : mEnabledFeatures(FeatureFlags::All()) {
  // Register the error lookup function for the module
  ttv::RegisterErrorToStringFunction(&ttv::social::SocialErrorToString);
  ttv::RegisterErrorCodeValueFunction(&ttv::social::GetSocialErrorCodeValues);

  mInternalData = std::make_shared<SocialAPIInternalData>();
}

std::string ttv::social::SocialAPI::GetModuleName() const {
  return kModule;
}

TTV_ErrorCode ttv::social::SocialAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  TTV_ASSERT(coreApi != nullptr);

  mCoreApi = coreApi;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::SocialAPI::SetListener(const std::shared_ptr<ISocialAPIListener>& listener) {
  if (mState != State::Uninitialized) {
    return TTV_EC_SHUT_DOWN;
  }

  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::SocialAPI::SetEnabledFeatures(const FeatureFlags& features) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Uninitialized, TTV_EC_ALREADY_INITIALIZED);

  mEnabledFeatures = features;

  return TTV_EC_SUCCESS;
}

void ttv::social::SocialAPI::CreateFriendListListener() {
  auto friendListListener = std::make_shared<FriendListListenerProxy>();
  mInternalData->mFriendListListener = friendListListener;

  friendListListener->mOnFriendInfoChangedFunc = [this](FriendList* source, const std::vector<Friend>& list) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();

    Invoke<ISocialAPIListener>([userId, &list](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialFriendInfoChanged(userId, list);
    });
  };

  friendListListener->mOnFriendshipChangedFunc = [this](FriendList* source, const std::vector<Friend>& added,
                                                   const std::vector<Friend>& removed) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();
    Invoke<ISocialAPIListener>([userId, &added, &removed](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialFriendshipChanged(userId, added, removed);
    });
  };

  friendListListener->mOnRealtimeFriendRequestReceivedFunc = [this](FriendList* source, const FriendRequest& request) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();
    Invoke<ISocialAPIListener>([userId, &request](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialRealtimeFriendRequestReceived(userId, request);
    });
  };

  friendListListener->mOnUnreadFriendRequestCountChangedFunc = [this](FriendList* source, uint32_t count) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();
    Invoke<ISocialAPIListener>([userId, count](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialUnreadFriendRequestCountChanged(userId, count);
    });
  };

  friendListListener->mOnFriendRequestRemovedFunc = [this](FriendList* source, UserId otherUserId,
                                                      FriendRequestRemovalReason reason) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();
    Invoke<ISocialAPIListener>([userId, otherUserId, reason](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialFriendRequestRemoved(userId, otherUserId, reason);
    });
  };
}

void ttv::social::SocialAPI::CreatePresenceListener() {
  auto presenceListener = std::make_shared<PresenceListenerProxy>();
  mInternalData->mPresenceListener = presenceListener;

  presenceListener->mOnSettingsChangedFunc = [this](Presence* source, const PresenceSettings& settings) {
    auto user = source->GetUser();
    if (user == nullptr) {
      return;
    }

    UserId userId = user->GetUserId();
    Invoke<ISocialAPIListener>([userId, &settings](const std::shared_ptr<ISocialAPIListener>& listener) {
      listener->SocialPresenceSettingsChanged(userId, settings);
    });
  };
}

TTV_ErrorCode ttv::social::SocialAPI::Initialize(const InitializeCallback& callback) {
  TTV_RETURN_ON_NULL(mCoreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;
    mCoreApiClient = std::make_shared<CoreApiClient>(this);

    mMainEventScheduler = mCoreApi->GetMainEventScheduler();
    mTaskRunner = std::make_shared<TaskRunner>("SocialAPI");
    mUserRepository = mCoreApi->GetUserRepository();
    CreateFriendListListener();
    CreatePresenceListener();

    ec = mCoreApi->RegisterClient(mCoreApiClient);
  }

  if (TTV_SUCCEEDED(ec)) {
    NotifyStateChange();
    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);
  if (TTV_SUCCEEDED(ec)) {
    // Cleanup data in users
    std::vector<std::shared_ptr<User>> users;
    mUserRepository->GetUsers(users);

    for (auto user : users) {
      CoreUserLoggedOut(user);
    }

    RegisterShutdownCallback(callback);
  }
  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  if (mUserRepository != nullptr) {
    mUserRepository->Update();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    case State::ShuttingDown: {
      if (CheckShutdown()) {
        CompleteShutdown();

        NotifyStateChange();
      }

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

bool ttv::social::SocialAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::social::SocialAPI::CompleteShutdown() {
  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  if (mCoreApiClient != nullptr) {
    if (mCoreApi != nullptr) {
      mCoreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  mCoreApi.reset();
  mUserRepository.reset();
  mInternalData->mFriendListListener.reset();
  mInternalData->mPresenceListener.reset();

  ModuleBase::CompleteShutdown();
}

TTV_ErrorCode ttv::social::SocialAPI::GetPresenceForUser(UserId userId, std::shared_ptr<Presence>& presence) {
  if (!mEnabledFeatures.presence) {
    return TTV_EC_FEATURE_DISABLED;
  }

  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_INVALID_USERID;
  }

  presence = user->GetComponentContainer()->GetComponent<Presence>();
  if (presence == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::SocialAPI::GetFriendListForUser(UserId userId, std::shared_ptr<FriendList>& friendList) {
  if (!mEnabledFeatures.friendList) {
    return TTV_EC_FEATURE_DISABLED;
  }

  std::shared_ptr<User> user = mUserRepository->GetUser(userId);
  if (user == nullptr) {
    return TTV_EC_INVALID_USERID;
  }

  friendList = user->GetComponentContainer()->GetComponent<FriendList>();
  if (friendList == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::social::SocialAPI::PostPresence(UserId userId, const PostPresenceCallback& callback) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->PostPresence(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::SetPresenceSessionAvailability(
  UserId userId, PresenceSessionAvailability availability) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->SetSessionAvailability(availability);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::AddWatchingActivity(
  UserId userId, ChannelId channelId, PresenceActivityToken& activityToken) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->AddWatchingActivity(channelId, activityToken);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::AddPlayingActivity(
  UserId userId, GameId gameId, const std::string& gameDisplayContext, PresenceActivityToken& activityToken) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->AddPlayingActivity(gameId, gameDisplayContext, activityToken);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::RemoveActivity(UserId userId, PresenceActivityToken activityToken) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->RemoveActivity(activityToken);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchPresenceSettings(
  UserId userId, const FetchPresenceSettingsCallback& callback) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->FetchSettings(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::SetPresenceSettings(
  UserId userId, const PresenceSettings& settings, const SetPresenceSettingsCallback& callback) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->SetPresenceSettings(settings, callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::SetAutomaticPresencePostingEnabled(UserId userId, bool enabled) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->SetAutomaticPresencePostingEnabled(enabled);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::GetAutomaticPresencePostingEnabled(UserId userId, bool& enabled) {
  std::shared_ptr<Presence> presence;
  TTV_ErrorCode ec = GetPresenceForUser(userId, presence);

  if (TTV_SUCCEEDED(ec)) {
    ec = presence->GetAutomaticPresencePostingEnabled(enabled);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchFriendList(UserId userId, const FetchFriendListCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->FetchFriendList(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::UpdateFriendship(
  UserId userId, UserId otherUserId, FriendAction action, const UpdateFriendshipCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->UpdateFriendship(otherUserId, action, callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchFriendRequests(UserId userId, const FetchFriendRequestsCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->FetchFriendRequests(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchUnreadFriendRequestCount(
  UserId userId, const FetchUnreadFriendRequestCountCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->FetchUnreadFriendRequestCount(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::MarkAllFriendRequestsRead(
  UserId userId, const MarkAllFriendRequestsReadCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->MarkAllFriendRequestsRead(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchFriendStatus(
  UserId userId, UserId otherUserId, const FetchFriendStatusCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->FetchFriendStatus(otherUserId, callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::FetchRecommendedFriends(
  UserId userId, const FetchRecommendedFriendsCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->FetchRecommendedFriends(callback);
  }

  return ec;
}

TTV_ErrorCode ttv::social::SocialAPI::DismissRecommendedFriend(
  UserId userId, UserId dismissUserid, const DismissRecommendedFriendCallback& callback) {
  std::shared_ptr<FriendList> friendList;
  TTV_ErrorCode ec = GetFriendListForUser(userId, friendList);

  if (TTV_SUCCEEDED(ec)) {
    ec = friendList->DismissRecommendedFriend(dismissUserid, callback);
  }

  return ec;
}

void ttv::social::SocialAPI::CoreUserLoggedIn(std::shared_ptr<User> user) {
  if (mState == State::Uninitialized || mState == State::ShuttingDown) {
    return;
  }

  // Update FriendList
  if (mEnabledFeatures.friendList) {
    std::shared_ptr<FriendList> friendList =
      user->GetComponentContainer()->GetComponentAs<FriendList>(FriendList::GetComponentName());
    if (friendList == nullptr) {
      friendList = std::make_shared<FriendList>(user, mEnabledFeatures);
      friendList->SetTaskRunner(mTaskRunner);
      friendList->SetUserRepository(mUserRepository);
      friendList->AddListener(mInternalData->mFriendListListener);

      user->GetComponentContainer()->SetComponent(FriendList::GetComponentName(), friendList);

      friendList->Initialize();
    }
  }

  // Update Presence
  if (mEnabledFeatures.presence) {
    std::shared_ptr<Presence> presence =
      user->GetComponentContainer()->GetComponentAs<Presence>(Presence::GetComponentName());
    if (presence == nullptr) {
      presence = std::make_shared<Presence>(user);
      presence->SetTaskRunner(mTaskRunner);
      presence->SetUserRepository(mUserRepository);
      presence->AddListener(mInternalData->mPresenceListener);

      user->GetComponentContainer()->SetComponent(Presence::GetComponentName(), presence);

      presence->Initialize();
    }
  }
}

void ttv::social::SocialAPI::CoreUserLoggedOut(std::shared_ptr<User> user) {
  // We don't need these components anymore
  auto container = user->GetComponentContainer();
  if (container != nullptr) {
    container->DisposeComponent(FriendList::GetComponentName());
    container->DisposeComponent(Presence::GetComponentName());
  }
}

ttv::social::SocialAPI::CoreApiClient::CoreApiClient(SocialAPI* api) : mApi(api) {}

void ttv::social::SocialAPI::CoreApiClient::CoreUserLoggedIn(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedIn(user);
}

void ttv::social::SocialAPI::CoreApiClient::CoreUserLoggedOut(std::shared_ptr<User> user) {
  mApi->CoreUserLoggedOut(user);
}
