/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialfriendrequeststask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"

#include <sstream>

namespace {
const char* kHostName = "https://api.twitch.tv";
const char* kApiVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// Fetch friend requests:
//   curl -X GET -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID/friends/requests" | python -m
//   json.tool GET /v5/users/:user_id/friends/requests
//
// Delete friend request notification count:
//   curl -X DELETE -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID/friends/notifications" --verbose
//   DELETE /v5/users/:user_id/friends/notifications
//
// Get the number of pending unread notifications:
//   curl -X GET -H -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID/friends/notifications" | python -m
//   json.tool GET /v5/users/:user_id/friends/notifications

ttv::social::SocialFriendRequestsTask::Result::Result() : total(0) {}

ttv::social::SocialFriendRequestsTask::SocialFriendRequestsTask(
  UserId userId, const std::string& authToken, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(callback),
      mAction(Action::Unset),
      mSortDirection(SortDirection::Descending),
      mUserId(userId),
      mLimit(0) {
  TTV_ASSERT(authToken.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SocialFriendRequestsTask created");
}

void ttv::social::SocialFriendRequestsTask::FetchRequests(
  uint32_t limit, SortDirection sortDirection, const std::string& cursor) {
  TTV_ASSERT(mAction == Action::Unset);
  mAction = Action::Fetch;
  mLimit = limit;
  mSortDirection = sortDirection;
  mCursor = cursor;
}

void ttv::social::SocialFriendRequestsTask::MarkAllRead() {
  TTV_ASSERT(mAction == Action::Unset);
  mAction = Action::MarkAllRead;
}

void ttv::social::SocialFriendRequestsTask::GetUnreadCount() {
  TTV_ASSERT(mAction == Action::Unset);
  mAction = Action::GetUnreadCount;
}

void ttv::social::SocialFriendRequestsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri uri;

  std::stringstream ss;
  ss << kHostName << "/kraken/users/" << mUserId << "/friends/";

  switch (mAction) {
    case Action::Fetch: {
      // SDK-778 Visage v5 API
      ss << "requests";
      requestInfo.httpReqType = HTTP_GET_REQUEST;
      break;
    }
    case Action::MarkAllRead: {
      // SDK-778 Visage v5 API
      ss << "notifications";
      requestInfo.httpReqType = HTTP_DELETE_REQUEST;
      break;
    }
    case Action::GetUnreadCount: {
      // SDK-778 Visage v5 API
      ss << "notifications";
      requestInfo.httpReqType = HTTP_GET_REQUEST;
      break;
    }
    default: {
      TTV_ASSERT(false);
      break;
    }
  }

  uri = ss.str();

  if (mAction == Action::Fetch) {
    if (mLimit > 0) {
      uri.SetParam("limit", mLimit);
    }

    switch (mSortDirection) {
      case SortDirection::Ascending: {
        uri.SetParam("direction", "ASC");
        break;
      }
      default: {
        uri.SetParam("direction", "DESC");
        break;
      }
    }

    if (mCursor != "") {
      uri.SetParam("cursor", mCursor);
    }
  }

  requestInfo.requestHeaders.emplace_back("Accept", kApiVersion);

  requestInfo.url = uri.GetUrl();
}

bool ttv::social::SocialFriendRequestsTask::ProcessHeaders(
  uint statusCode, const std::map<std::string, std::string>& /*headers*/) {
  if (statusCode >= 200 && statusCode < 300) {
    switch (mAction) {
      case Action::MarkAllRead: {
        mResult = std::make_shared<Result>();
        mResult->action = mAction;
        break;
      }
      default: { break; }
    }
  } else {
    mTaskStatus = TTV_EC_API_REQUEST_FAILED;
  }

  return mAction == Action::Fetch || mAction == Action::GetUnreadCount;
}

void ttv::social::SocialFriendRequestsTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside SocialFriendRequestsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();
    mResult->action = mAction;
    mResult->direction = mSortDirection;

    switch (mAction) {
      case Action::Fetch: {
        const auto& jTotal = jsonVal["_total"];
        const auto& jCursor = jsonVal["_cursor"];
        const auto& jRequests = jsonVal["requests"];

        if (jTotal.isNull() || !jTotal.isNumeric()) {
          ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing JSON: '_total' is not valid");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          break;
        }
        mResult->total = static_cast<int32_t>(jTotal.asInt());

        if (jCursor.isNull() || !jCursor.isString()) {
          ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing JSON: '_cursor' is not valid");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          break;
        }
        mResult->cursor = jCursor.asString();

        bool succeeded = ParseFriendRequestListJson(jRequests);
        if (!succeeded) {
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        }

        break;
      }
      case Action::GetUnreadCount: {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;

        if (!jsonVal.isNull() && jsonVal.isObject()) {
          const auto& jCount = jsonVal["count"];
          if (jCount.isNumeric()) {
            mResult->total = static_cast<int32_t>(jCount.asUInt());
            mTaskStatus = TTV_EC_SUCCESS;
          }
        }

        break;
      }
      default: {
        TTV_ASSERT(false);
        break;
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

bool ttv::social::SocialFriendRequestsTask::ParseFriendRequestListJson(const json::Value& jArray) {
  if (jArray.isNull() || !jArray.isArray()) {
    return false;
  }

  for (size_t i = 0; i < jArray.size(); ++i) {
    const auto& jEntry = jArray[i];
    if (!jEntry.isNull() && jEntry.isObject()) {
      FriendRequestEntry entry;

      ParseTimestamp(jEntry["requested_at"], entry.requestedTimestamp);
      if (!json::ToObject(jEntry["user"], entry.userInfo)) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing 'user' JSON");
        return false;
      }

      mResult->requests.push_back(entry);
    }
  }

  return true;
}

void ttv::social::SocialFriendRequestsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
