/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialgetfriendspresencetask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/social/internal/socialhelpers.h"

namespace {
const char* kHostName = "https://api.twitch.tv";
}

// curl -X GET -H "Authorization: OAuth $OAUTH" -H "Client-ID: $CLIENTID"
// https://api.twitch.tv/v5/users/$USERID/status/friends | python -m json.tool GET /v5/users/:user_id/status/friends

ttv::social::SocialGetFriendsPresenceTask::SocialGetFriendsPresenceTask(
  UserId userId, const std::string& authToken, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(callback), mUserId(userId) {
  TTV_ASSERT(userId != 0);
  TTV_ASSERT(authToken.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SocialGetFriendsPresenceTask created");
}

void ttv::social::SocialGetFriendsPresenceTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.requestHeaders.emplace_back(ttv::HttpParam("Accept", "application/json"));

  requestInfo.httpReqType = HTTP_GET_REQUEST;

  std::stringstream urlStream;
  // SDK-778 Visage v5 API
  urlStream << kHostName << "/v5/users/" << mUserId << "/status/friends";
  requestInfo.url = urlStream.str();
}

bool ttv::social::SocialGetFriendsPresenceTask::ProcessHeaders(
  uint statusCode, const std::map<std::string, std::string>& /*headers*/) {
  if (statusCode >= 200 && statusCode < 300) {
    return true;
  } else if (statusCode == 401) {
    mTaskStatus = TTV_EC_AUTHENTICATION;
  } else {
    mTaskStatus = TTV_EC_API_REQUEST_FAILED;
  }

  return false;
}

void ttv::social::SocialGetFriendsPresenceTask::ProcessResponse(
  uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside SocialGetFriendsPresenceTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();

    const auto& jData = jsonVal["data"];
    if (jData.isArray()) {
      if (!jData.isNull()) {
        for (const auto& jEntry : jData) {
          if (!jEntry.isNull() && jEntry.isObject()) {
            PresenceEntry info;

            if (!json::ToObject(jEntry["user"], info.friendData.userInfo)) {
              continue;
            }

            const auto& jAvailability = jEntry["availability"];
            if (jAvailability.isNull() || !jAvailability.isString()) {
              continue;
            }

            if (!ParsePresenceUserAvailability(jAvailability.asString(), info.friendData.presenceStatus.availability)) {
              ttv::trace::Message(
                GetTaskName(), MessageLevel::Error, "Unhandled 'availability': %s", jAvailability.asString().c_str());
            }

            const auto& jIndex = jEntry["index"];
            if (jIndex.isNull() || !jIndex.isNumeric()) {
              continue;
            }
            info.presenceUpdateIndex = static_cast<uint64_t>(jIndex.asInt());

            const auto& jUpdated = jEntry["updated_at"];
            ParseTimestamp(jUpdated, info.friendData.presenceStatus.lastUpdate);

            const auto& jActivity = jEntry["activity"];
            if (!jActivity.isNull() && jActivity.isObject()) {
              CreatePresenceActivity(jActivity, info.friendData.presenceStatus.activity);
            }

            mResult->presenceList.push_back(std::move(info));
          }
        }
      }
    } else {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::social::SocialGetFriendsPresenceTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
