/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialjson.h"

#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"

namespace {
using namespace ttv;
using namespace ttv::social;

TTV_ErrorCode GenerateBroadcastingActivityJson(const BroadcastingActivity& activity, json::Value& value) {
  if (activity.channelId == 0) {
    return TTV_EC_INVALID_INSTANCE;
  }
  value["type"] = "broadcasting";
  value["channel_id"] = std::to_string(activity.channelId);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode GenerateWatchingActivityJson(const WatchingActivity& activity, json::Value& value) {
  if (activity.channelId == 0) {
    return TTV_EC_INVALID_INSTANCE;
  }
  value["type"] = "watching";
  value["channel_id"] = std::to_string(activity.channelId);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode GeneratePlayingActivityJson(const PlayingActivity& activity, json::Value& value) {
  if (activity.gameId == 0) {
    return TTV_EC_INVALID_INSTANCE;
  }
  value["type"] = "playing";
  value["game_id"] = std::to_string(activity.gameId);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode CreateBroadcastingActivity(const json::Value& jObject, std::unique_ptr<PresenceActivity>& activity) {
  auto broadcastingActivity = std::make_unique<BroadcastingActivity>();
  if (!ParseChannelId(jObject["channel_id"], broadcastingActivity->channelId)) {
    return TTV_EC_INVALID_JSON;
  }

  const auto& jChannelLogin = jObject["channel_login"];
  if (!jChannelLogin.isNull() && jChannelLogin.isString()) {
    broadcastingActivity->channelLogin = jChannelLogin.asString();
  }

  const auto& jChannelDisplayName = jObject["channel_display_name"];
  if (!jChannelDisplayName.isNull() && jChannelDisplayName.isString()) {
    broadcastingActivity->channelDisplayName = jChannelDisplayName.asString();
  }

  ParseGameId(jObject["game_id"], broadcastingActivity->gameId);

  const auto& jGameName = jObject["game"];
  if (!jGameName.isNull() && jGameName.isString()) {
    broadcastingActivity->gameName = jGameName.asString();
  }

  activity = std::move(broadcastingActivity);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode CreateWatchingActivity(const json::Value& jObject, std::unique_ptr<PresenceActivity>& activity) {
  auto watchingActivity = std::make_unique<WatchingActivity>();

  if (!ParseChannelId(jObject["channel_id"], watchingActivity->channelId)) {
    return TTV_EC_INVALID_JSON;
  }

  const auto& jChannelLogin = jObject["channel_login"];
  if (!jChannelLogin.isNull() && jChannelLogin.isString()) {
    watchingActivity->channelLogin = jChannelLogin.asString();
  }

  const auto& jChannelDisplayName = jObject["channel_display_name"];
  if (!jChannelDisplayName.isNull() && jChannelDisplayName.isString()) {
    watchingActivity->channelDisplayName = jChannelDisplayName.asString();
  }

  ParseChannelId(jObject["hosted_channel_id"], watchingActivity->hostedChannelId);

  const auto& jHostedChannelLogin = jObject["hosted_channel_login"];
  if (!jHostedChannelLogin.isNull() && jHostedChannelLogin.isString()) {
    watchingActivity->hostedChannelLogin = jHostedChannelLogin.asString();
  }

  const auto& jHostedChannelDisplayName = jObject["hosted_channel_display_name"];
  if (!jHostedChannelDisplayName.isNull() && jHostedChannelDisplayName.isString()) {
    watchingActivity->hostedChannelDisplayName = jHostedChannelDisplayName.asString();
  }

  ParseGameId(jObject["game_id"], watchingActivity->gameId);

  const auto& jGameName = jObject["game"];
  if (!jGameName.isNull() && jGameName.isString()) {
    watchingActivity->gameName = jGameName.asString();
  }

  activity = std::move(watchingActivity);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode CreatePlayingActivity(const json::Value& jObject, std::unique_ptr<PresenceActivity>& activity) {
  auto playingActivity = std::make_unique<PlayingActivity>();
  if (!ParseGameId(jObject["game_id"], playingActivity->gameId)) {
    return TTV_EC_INVALID_JSON;
  }

  const auto& jGameName = jObject["game"];
  if (!jGameName.isNull() && jGameName.isString()) {
    playingActivity->gameName = jGameName.asString();
  }

  const auto& jGameDisplayContext = jObject["game_display_context"];
  if (!jGameDisplayContext.isNull() && jGameDisplayContext.isString()) {
    playingActivity->gameDisplayContext = jGameDisplayContext.asString();
  }

  activity = std::move(playingActivity);

  return TTV_EC_SUCCESS;
}
}  // namespace

TTV_ErrorCode ttv::social::GenerateActivityJson(const ttv::social::PresenceActivity& activity, json::Value& value) {
  switch (activity.GetType()) {
    case PresenceActivity::Type::Broadcasting: {
      return GenerateBroadcastingActivityJson(static_cast<const BroadcastingActivity&>(activity), value);
    }
    case PresenceActivity::Type::Watching: {
      return GenerateWatchingActivityJson(static_cast<const WatchingActivity&>(activity), value);
    }
    case PresenceActivity::Type::Playing: {
      return GeneratePlayingActivityJson(static_cast<const PlayingActivity&>(activity), value);
    }
    default: { return TTV_EC_UNIMPLEMENTED; }
  }
}

TTV_ErrorCode ttv::social::CreatePresenceActivity(
  const json::Value& jObject, std::unique_ptr<PresenceActivity>& activity) {
  if (jObject.isNull() || !jObject.isObject()) {
    return TTV_EC_INVALID_JSON;
  }

  const auto& jType = jObject["type"];
  if (jType.isNull() || !jType.isString()) {
    return TTV_EC_INVALID_JSON;
  }

  auto type = jType.asString();
  if (type == "broadcasting") {
    return CreateBroadcastingActivity(jObject, activity);
  } else if (type == "watching") {
    return CreateWatchingActivity(jObject, activity);
  } else if (type == "playing") {
    return CreatePlayingActivity(jObject, activity);
  } else {
    return TTV_EC_INVALID_JSON;
  }
}
