/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialpresencesettingstask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/social/internal/socialhelpers.h"

namespace {
const char* kHostName = "https://api.twitch.tv";
}

// curl -X GET -H 'Authorization: OAuth <access_token>' -H 'Client-Id: $CLIENTID'
// https://api.twitch.tv/v5/users/$USERID/status/settings | python -m json.tool curl -X POST -H 'Authorization: OAuth
// <access_token>' -H 'Client-Id: $CLIENTID' -d 'availability_override=away&share_activity=true'
// https://api.twitch.tv/v5/users/$USERID/status/settings | python -m json.tool GET /v5/users/:user_id/status/settings
// POST /v5/users/:user_id/status/settings

ttv::social::SocialPresenceSettingsTask::SocialPresenceSettingsTask(
  UserId userId, const std::string& authToken, bool set, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(callback), mUserId(userId), mSet(set) {
  TTV_ASSERT(userId != 0);
  TTV_ASSERT(authToken.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SocialPresenceSettingsTask created");
}

void ttv::social::SocialPresenceSettingsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  if (mSet) {
    json::Value root;

    switch (mSettings.availabilityOverride) {
      case PresenceSettings::AvailabilityOverride::None: {
        root["availability_override"] = "";
        break;
      }
      case PresenceSettings::AvailabilityOverride::Away: {
        root["availability_override"] = "away";
        break;
      }
      case PresenceSettings::AvailabilityOverride::Busy: {
        root["availability_override"] = "busy";
        break;
      }
      case PresenceSettings::AvailabilityOverride::Offline: {
        root["availability_override"] = "offline";
        break;
      }
      default: {
        ttv::trace::Message(GetTaskName(), MessageLevel::Debug, "Unrecognized PresenceUserAvailability value");
        break;
      }
    }

    root["share_activity"] = mSettings.shareActivity;

    json::FastWriter writer;
    requestInfo.requestBody = writer.write(root);

    requestInfo.requestHeaders.emplace_back("Content-Type", "application/json");
    // SDK-778 Visage v5 API
    requestInfo.httpReqType = HTTP_POST_REQUEST;
  } else {
    requestInfo.requestHeaders.emplace_back("Accept", "application/json");
    // SDK-778 Visage v5 API
    requestInfo.httpReqType = HTTP_GET_REQUEST;
  }

  std::stringstream urlStream;
  urlStream << kHostName << "/v5/users/" << mUserId << "/status/settings";
  requestInfo.url = urlStream.str();
}

void ttv::social::SocialPresenceSettingsTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside SocialPresenceSettingsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jAvailabilityOverride = jsonVal["availability_override"];
    const auto& jShareActivity = jsonVal["share_activity"];
    if (!jAvailabilityOverride.isNull() && jAvailabilityOverride.isString() && !jShareActivity.isNull() &&
        jShareActivity.isBool()) {
      mResult = std::make_shared<Result>();

      if (jAvailabilityOverride.asString() == "") {
        mResult->settings.availabilityOverride = PresenceSettings::AvailabilityOverride::None;
      } else if (jAvailabilityOverride.asString() == "away") {
        mResult->settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Away;
      } else if (jAvailabilityOverride.asString() == "busy") {
        mResult->settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Busy;
      } else if (jAvailabilityOverride.asString() == "offline") {
        mResult->settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Offline;
      } else {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Unrecognized availability override: %s",
          jAvailabilityOverride.asString().c_str());
      }

      mResult->settings.shareActivity = jShareActivity.asBool();
    } else {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::social::SocialPresenceSettingsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
