/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialrecommendedfriendstask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"

#include <sstream>

namespace {
const char* kHostName = "https://api.twitch.tv";
const char* kApiVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// Get the current list of recommended friends:
//     curl -X GET -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//     application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID/friends/recommendations" | python
//     -m json.tool GET /v5/users/:user_id/friends/recommendations
//
// Dismiss a recommended friend:
//     curl -X DELETE -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//     application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/recommendations/$USERID1"
//     | python -m json.tool DELETE /v5/users/:user_id/friends/recommendations/:target_id

ttv::social::SocialRecommendedFriendsTask::Result::Result() : action(Action::Unset) {}

ttv::social::SocialRecommendedFriendsTask::SocialRecommendedFriendsTask(
  UserId userId, const std::string& authToken, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(callback),
      mUserId(userId),
      mRecommendedUserId(0),
      mAction(Action::Unset) {
  TTV_ASSERT(authToken.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SocialRecommendedFriendsTask created");
}

void ttv::social::SocialRecommendedFriendsTask::Fetch() {
  mAction = Action::Fetch;
}

void ttv::social::SocialRecommendedFriendsTask::Dismiss(UserId userId) {
  mAction = Action::Dismiss;
  mRecommendedUserId = userId;
}

void ttv::social::SocialRecommendedFriendsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.requestHeaders.emplace_back(ttv::HttpParam("Accept", "application/json"));

  Uri uri;

  std::stringstream ss;
  ss << kHostName << "/kraken/users/" << mUserId << "/friends/recommendations";

  switch (mAction) {
    case Action::Fetch: {
      // SDK-778 Visage v5 API
      requestInfo.httpReqType = HTTP_GET_REQUEST;
      break;
    }
    case Action::Dismiss: {
      ss << "/" << mRecommendedUserId;
      // SDK-778 Visage v5 API
      requestInfo.httpReqType = HTTP_DELETE_REQUEST;
      break;
    }
    default: {
      TTV_ASSERT(false);
      break;
    }
  }

  uri = ss.str();

  requestInfo.requestHeaders.push_back(HttpParam("Accept", kApiVersion));

  requestInfo.url = uri.GetUrl();
}

bool ttv::social::SocialRecommendedFriendsTask::ProcessHeaders(
  uint statusCode, const std::map<std::string, std::string>& /*headers*/) {
  if (statusCode >= 200 && statusCode < 300) {
    switch (mAction) {
      case Action::Dismiss: {
        mResult = std::make_shared<Result>();
        mResult->action = mAction;
        break;
      }
      default: { break; }
    }
  } else if (statusCode == 401) {
    mTaskStatus = TTV_EC_AUTHENTICATION;
  } else {
    mTaskStatus = TTV_EC_API_REQUEST_FAILED;
  }

  return mAction == Action::Fetch;
}

void ttv::social::SocialRecommendedFriendsTask::ProcessResponse(
  uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside SocialRecommendedFriendsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();
    mResult->action = mAction;

    switch (mAction) {
      case Action::Fetch: {
        const auto& jRecommended = jsonVal["recommendations"];

        if (jRecommended.isNull() || !jRecommended.isArray()) {
          ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing JSON: 'recommended' is not valid");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          break;
        }

        bool succeeded = ParseRecommendedFriendListJson(jRecommended);
        if (!succeeded) {
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        }

        break;
      }
      default: {
        TTV_ASSERT(false);
        break;
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

bool ttv::social::SocialRecommendedFriendsTask::ParseRecommendedFriendListJson(const json::Value& jArray) {
  if (jArray.isNull() || !jArray.isArray()) {
    return false;
  }

  for (size_t i = 0; i < jArray.size(); ++i) {
    const auto& jEntry = jArray[i];
    if (!jEntry.isNull() && jEntry.isObject()) {
      RecommendedFriendEntry entry;

      if (!json::ToObject(jEntry["user"], entry.userInfo)) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing 'user' JSON");
        return false;
      }

      mResult->recommendedFriends.push_back(entry);
    }
  }

  return true;
}

void ttv::social::SocialRecommendedFriendsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
