/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/internal/task/socialupdatefriendtask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"

#include <sstream>

namespace {
const char* kHostName = "https://api.twitch.tv";
const char* kApiVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// Send a friend request
//   curl -X PUT -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/requests/$USERID2" --verbose
//   PUT /v5/users/:user_id/friends/requests/:target_id
//
// Accept a friend request
//   curl -X PUT -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH2" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/relationships/$USERID2"
//   --verbose PUT /v5/users/:user_id/friends/relationships/:target_id
//
// Reject a friend request
//   curl -X DELETE -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH2" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/requests/$USERID2" --verbose
//   DELETE /v5/users/:user_id/friends/requests/:target_id
//
// Remove a friend
//   curl -X DELETE -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/relationships/$USERID2"
//   --verbose DELETE /v5/users/:user_id/friends/relationships/:target_id
//
// Get friend status
//   curl -X GET -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH1" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/users/$USERID1/friends/relationships/$USERID2" |
//   python -m json.tool GET /v5/users/:user_id/friends/relationships/:target_id

ttv::social::SocialUpdateFriendTask::Result::Result()
    : result(UpdateFriendResult::RequestNotAllowed), status(FriendStatus::Unknown) {}

ttv::social::SocialUpdateFriendTask::SocialUpdateFriendTask(
  UserId userId, const std::string& authToken, UserId friendUserId, Action action, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(callback),
      mUserId(userId),
      mFriendUserId(friendUserId),
      mAction(action) {
  TTV_ASSERT(authToken.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "SocialUpdateFriendTask created");
}

void ttv::social::SocialUpdateFriendTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << "/kraken/users/" << mUserId << "/friends/";

  switch (mAction) {
    case Action::SendRequest: {
      // SDK-778 Visage v5 API
      ss << "requests/" << mFriendUserId;
      requestInfo.httpReqType = HTTP_PUT_REQUEST;
      break;
    }
    case Action::AcceptRequest: {
      // SDK-778 Visage v5 API
      ss << "relationships/" << mFriendUserId;
      requestInfo.httpReqType = HTTP_PUT_REQUEST;
      break;
    }
    case Action::RejectRequest: {
      // SDK-778 Visage v5 API
      ss << "requests/" << mFriendUserId;
      requestInfo.httpReqType = HTTP_DELETE_REQUEST;
      break;
    }
    case Action::DeleteFriend: {
      // SDK-778 Visage v5 API
      ss << "relationships/" << mFriendUserId;
      requestInfo.httpReqType = HTTP_DELETE_REQUEST;
      break;
    }
    case Action::GetStatus: {
      // SDK-778 Visage v5 API
      ss << "relationships/" << mFriendUserId;
      requestInfo.httpReqType = HTTP_GET_REQUEST;
      break;
    }
    default: {
      TTV_ASSERT(false);
      break;
    }
  }

  requestInfo.requestHeaders.emplace_back("Accept", kApiVersion);

  requestInfo.url = ss.str();
}

bool ttv::social::SocialUpdateFriendTask::ProcessHeaders(
  uint statusCode, const std::map<std::string, std::string>& /*headers*/) {
  // https://docs.google.com/document/d/1IhbPQAKtwouqco6MTJVGkUUVkeD4J8LlovISQbA9l3Y

  if (!mAborted) {
    mResult = std::make_shared<Result>();

    if (statusCode == 401) {
      mTaskStatus = TTV_EC_AUTHENTICATION;
    } else {
      mTaskStatus = TTV_EC_SUCCESS;

      if (statusCode == 403) {
        mResult->result = UpdateFriendResult::RequestNotAllowed;
      } else {
        switch (mAction) {
          case Action::SendRequest: {
            if (statusCode >= 200 && statusCode < 300) {
              mResult->result = UpdateFriendResult::RequestSent;
            } else if (statusCode == 409) {
              mResult->result = UpdateFriendResult::AlreadyExists;
            } else {
              // Unexpected response
              mTaskStatus = TTV_EC_API_REQUEST_FAILED;
              mResult.reset();
            }

            break;
          }
          case Action::AcceptRequest: {
            if (statusCode >= 200 && statusCode < 300) {
              mResult->result = UpdateFriendResult::RequestAccepted;
            } else if (statusCode == 404) {
              mResult->result = UpdateFriendResult::RequestNotFound;
            } else {
              // Unexpected response
              mTaskStatus = TTV_EC_API_REQUEST_FAILED;
              mResult.reset();
            }

            break;
          }
          case Action::RejectRequest: {
            if (statusCode >= 200 && statusCode < 300) {
              mResult->result = UpdateFriendResult::RequestRejected;
            } else if (statusCode == 404) {
              // NOTE: It doesn't seem like this is returned
              mResult->result = UpdateFriendResult::RequestNotFound;
            } else {
              // Unexpected response
              mTaskStatus = TTV_EC_API_REQUEST_FAILED;
              mResult.reset();
            }

            break;
          }
          case Action::DeleteFriend: {
            if (statusCode >= 200 && statusCode < 300) {
              mResult->result = UpdateFriendResult::FriendDeleted;
            } else if (statusCode == 404) {
              mResult->result = UpdateFriendResult::FriendNotFound;
            } else {
              // Unexpected response
              mTaskStatus = TTV_EC_API_REQUEST_FAILED;
              mResult.reset();
            }

            break;
          }
          case Action::GetStatus: {
            if (statusCode >= 200 && statusCode < 300) {
              // NOTE: We parse the body in ProcessResponse
            } else {
              // Unexpected response
              mTaskStatus = TTV_EC_API_REQUEST_FAILED;
              mResult.reset();
            }

            break;
          }
        }
      }
    }
  }

  return mAction == Action::GetStatus;
}

void ttv::social::SocialUpdateFriendTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  TTV_ASSERT(mAction == Action::GetStatus);

  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside SocialUpdateFriendTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jStatus = jsonVal["status"];
    if (!jStatus.isNull() && jStatus.isString()) {
      std::string status = jStatus.asString();

      mResult = std::make_shared<Result>();

      if (status == "no_relation") {
        mResult->status = FriendStatus::NoRelation;
      } else if (status == "blocked") {
        mResult->status = FriendStatus::Blocked;
      } else if (status == "blocks") {
        mResult->status = FriendStatus::Blocks;
      } else if (status == "sent_request") {
        mResult->status = FriendStatus::SentRequest;
      } else if (status == "requested_by") {
        mResult->status = FriendStatus::ReceivedRequest;
      } else if (status == "friends") {
        mResult->status = FriendStatus::Friends;
      } else {
        mResult->status = FriendStatus::Unknown;
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "SocialUpdateFriendTask::ProcessResponse - unhandled friend status: %s", status.c_str());
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::social::SocialUpdateFriendTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
