/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/socialapitest.h"
#include "testutilities.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/social/socialapi.h"
#include "twitchsdk/social/sociallistener.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::social;
using namespace ttv::social::test;

TEST_F(SocialApiTest, Initalization) {
  TestModuleInitialization();
}

TEST_F(SocialApiTest, SetPresenceSettings) {
  TestModuleInitialization();

  UserInfo info = mUserTestUtility.DefaultUserInfo();
  LogIn(mCoreApi, "auth_token", info);

  auto response =
    mHttpRequest
      ->AddResponse(std::string("https://api.twitch.tv/v5/users/") + std::to_string(info.userId) + "/status/settings")
      .SetType(ttv::HttpRequestType::HTTP_POST_REQUEST)
      .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
      .AddRequestHeader("Authorization", std::string("OAuth ") + "auth_token")
      .SetResponseHandler([](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& body) {
        json::Reader reader;
        json::Value root;

        EXPECT_TRUE(reader.parse(body, root));

        EXPECT_TRUE(root["share_activity"].asBool());

        MockResponse::MockResponseContent responseContent;
        std::string responseString =
          "{\"availability_override\": \"\", \"is_invisible\": false, \"share_activity\": true}";
        responseContent.body.assign(responseString.begin(), responseString.end());
        return responseContent;
      })
      .Done();

  PresenceSettings settings;
  settings.availabilityOverride = PresenceSettings::AvailabilityOverride::None;
  settings.shareActivity = true;

  bool callbackCalled = false;
  TTV_ErrorCode ec = mSocialApi->SetPresenceSettings(info.userId, settings, [&callbackCalled](TTV_ErrorCode ec) {
    EXPECT_TRUE(TTV_SUCCEEDED(ec));
    callbackCalled = true;
  });

  EXPECT_TRUE(TTV_SUCCEEDED(ec));

  EXPECT_TRUE(WaitUntilResult(2000, [&callbackCalled]() { return callbackCalled; }));
}

TEST_F(SocialApiTest, FriendListRequests) {
  TestModuleInitialization();

  UserInfo info = mUserTestUtility.DefaultUserInfo();

  auto friendListFetch =
    mHttpRequest
      ->AddResponse(std::string("https://api.twitch.tv/v5/users/") + std::to_string(info.userId) + "/status/friends")
      .SetResponseBodyFromFile("social/socialfriendspresencetask.json")
      .Done();
  auto friendRequestFetch = mHttpRequest
                              ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") +
                                            std::to_string(info.userId) + "/friends/requests")
                              .StrictRequestParams(false)
                              .SetResponseBodyFromFile("social/socialfriendrequeststask_fetch.json")
                              .Done();
  auto friendRequestUnreadCountFetch = mHttpRequest
                                         ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") +
                                                       std::to_string(info.userId) + "/friends/notifications")
                                         .SetResponseBodyFromFile("social/socialfriendrequeststask_getunreadcount.json")
                                         .Done();
  auto friendRecommendationsFetch = mHttpRequest
                                      ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") +
                                                    std::to_string(info.userId) + "/friends/recommendations")
                                      .SetResponseBodyFromFile("social/socialrecommendedfriendstask.json")
                                      .Done();

  LogIn(mCoreApi, "auth_token", info);

  // Make explicit fetches so we don't have to wait on the initial timers.
  mSocialApi->FetchFriendList(info.userId, nullptr);
  mSocialApi->FetchFriendRequests(info.userId, nullptr);
  mSocialApi->FetchUnreadFriendRequestCount(info.userId, nullptr);
  mSocialApi->FetchRecommendedFriends(info.userId, nullptr);

  // Make sure we don't go over the expected number of requests for each request.
  ASSERT_FALSE(WaitUntilResult(
    5000, [&friendListFetch, &friendRequestFetch, &friendRequestUnreadCountFetch, &friendRecommendationsFetch]() {
      if (friendListFetch->GetRequestCount() > 2) {
        // Friend list may fetch up to two times (because we may need to re-fetch after we've finished subscribing to
        // PubSub).
        return true;
      } else if (friendRequestFetch->GetRequestCount() > 2) {
        // Friend requests also may fetch up to two times(because we may need to re-fetch after we've finished
        // subscribing to PubSub).
        return true;
      } else if (friendRequestUnreadCountFetch->GetRequestCount() > 1) {
        // Friend request unread count fetch should only happen once.
        return true;
      } else if (friendRecommendationsFetch->GetRequestCount() > 1) {
        // Friend recommendations should only happen once.
        return true;
      } else {
        return false;
      }
    }));

  // Make sure we got each request at least once.
  ASSERT_GT(friendListFetch->GetRequestCount(), 0);
  ASSERT_GT(friendRequestFetch->GetRequestCount(), 0);
  ASSERT_GT(friendRequestUnreadCountFetch->GetRequestCount(), 0);
  ASSERT_GT(friendRecommendationsFetch->GetRequestCount(), 0);
}
