/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "pubsubtestutility.h"
#include "socialtestutilities.h"
#include "testutilities.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/social/internal/friendlist.h"
#include "twitchsdk/social/internal/socialhelpers.h"
#include "twitchsdk/social/internal/task/socialgetfriendspresencetask.h"
#include "usertestutility.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;

namespace {
class FriendListTest : public SdkBaseTest {
 protected:
  FriendListTest() : mSampleData("social/samplefrienddata.json") {}

  virtual void SetUpStubs() override {
    SdkBaseTest::SetUpStubs();

    PopulateBasicData();
  }

  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    auto userRepository = CreateUserRepository();
    InitializeComponent(userRepository);

    UserInfo userInfo = mUserTestUtility.DefaultUserInfo();
    auto user = GetUserRepository()->RegisterUser(userInfo.userId);
    user->SetOAuthToken(std::make_shared<OAuthToken>("auth_token"));
    user->SetUserInfo(userInfo);

    mPubSubTestUtility.SetUpComponents(CreateTaskRunner(), user);
    InitializeComponent(mPubSubTestUtility.GetPubSubClient());

    mFriendList = std::make_shared<ttv::social::FriendList>(user, ttv::social::FeatureFlags::All());
    mFriendList->SetTaskRunner(CreateTaskRunner());
    AddComponent(mFriendList);

    mFriendListListenerProxy = std::make_shared<ttv::social::FriendListListenerProxy>();
    mFriendList->AddListener(mFriendListListenerProxy);

    VerifyBasicData();
  }

  virtual void TearDownComponents() override {
    mPubSubTestUtility.TearDownComponents();

    SdkBaseTest::TearDownComponents();
  }

  virtual void TearDownStubs() override { SdkBaseTest::TearDownStubs(); }

  void PopulateFriendList() {
    std::string userIdString = std::to_string(mUserTestUtility.DefaultUserInfo().userId);
    json::FastWriter writer;
    mHttpRequest->AddResponse("https://api.twitch.tv/v5/users/" + userIdString + "/status/friends")
      .SetResponseBody(writer.write(FriendsPresenceResponse()))
      .Done();

    mHttpRequest
      ->AddResponse("https://api.twitch.tv/kraken/users/" + userIdString + "/friends/requests?direction=DESC&limit=500")
      .SetResponseBody(writer.write(FriendRequestFetchResponse()))
      .Done();
  }

  json::Value FriendsPresenceResponse() {
    json::Value presences;

    // The real endpoint includes the users own status, which we end up filtering out.
    // We add it here, to test that it gets filtered out by the friend list.
    auto selfInfo = mUserTestUtility.DefaultUserInfo();
    presences.append(mSampleData.GeneratePresenceJson(selfInfo, mSampleData.GenerateRandomTimestamp(),
      mSampleData.GenerateRandomTimestamp(), 0, ttv::social::PresenceUserAvailability::Online,
      mSampleData.GetActivityJson(0)));

    for (const auto& presenceEntry : mFriendPresenceEntries) {
      presences.append(presenceEntry);
    }

    json::Value response;
    response["data"] = presences;

    return response;
  }

  json::Value FriendRequestFetchResponse() {
    json::Value requests;
    for (const auto& friendRequest : mFriendRequests) {
      json::Value requestEntryJson;
      requestEntryJson["requested_at"] = friendRequest.requestedTimestamp;
      requestEntryJson["user"] = mUserTestUtility.JsonForUserInfoKraken(friendRequest.userInfo);

      requests.append(std::move(requestEntryJson));
    }

    json::Value root;
    root["_total"] = static_cast<uint64_t>(mFriendRequests.size());
    root["_cursor"] = "SomeArbitraryString";
    root["requests"] = std::move(requests);

    return root;
  }

  json::Value RecommendedFriendFetchResponse() {
    json::Value recommendedFriends;
    for (const auto& recommendedInfo : mRecommendedFriends) {
      json::Value entry;
      entry["user"] = mUserTestUtility.JsonForUserInfoKraken(recommendedInfo);
      recommendedFriends.append(std::move(entry));
    }

    json::Value root;
    root["recommendations"] = recommendedFriends;
    return root;
  }

  void PopulateBasicData() {
    auto friendInfo1 = mSampleData.GetFriendInfo(0);
    auto friendInfo2 = mSampleData.GetFriendInfo(1);
    auto friendInfo3 = mSampleData.GetFriendInfo(2);

    mFriendPresenceEntries.push_back(mSampleData.GeneratePresenceJson(friendInfo1,
      mSampleData.GenerateRandomTimestamp(), mSampleData.GenerateRandomTimestamp(), 0,
      ttv::social::PresenceUserAvailability::Online, mSampleData.GetActivityJson(0)));
    mFriendPresenceEntries.push_back(
      mSampleData.GeneratePresenceJson(friendInfo2, mSampleData.GenerateRandomTimestamp(),
        mSampleData.GenerateRandomTimestamp(), 0, ttv::social::PresenceUserAvailability::Offline, json::nullValue));
    mFriendPresenceEntries.push_back(mSampleData.GeneratePresenceJson(friendInfo3,
      mSampleData.GenerateRandomTimestamp(), mSampleData.GenerateRandomTimestamp(), 0,
      ttv::social::PresenceUserAvailability::Away, mSampleData.GetActivityJson(1)));

    mFriendRequests.push_back(
      mSampleData.GenerateFriendRequestEntry(mSampleData.GetFriendInfo(3), mSampleData.GenerateRandomTimestamp()));
    mFriendRequests.push_back(
      mSampleData.GenerateFriendRequestEntry(mSampleData.GetFriendInfo(4), mSampleData.GenerateRandomTimestamp()));

    PopulateFriendList();
  }

  void VerifyBasicData() {
    InitializeComponent(mFriendList);

    std::vector<ttv::social::Friend> friends;
    bool fetchComplete = false;
    TTV_ErrorCode ec = mFriendList->FetchFriendList(
      [&friends, &fetchComplete](TTV_ErrorCode ec, const std::vector<ttv::social::Friend>& fetchedFriends) {
        EXPECT_TRUE(TTV_SUCCEEDED(ec));
        friends = fetchedFriends;
        fetchComplete = true;
      });

    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    WaitUntilResult(1000, [&fetchComplete]() { return fetchComplete; });

    ASSERT_EQ(friends.size(), 3);

    json::Value presenceJsonToCheck;
    auto compareEntries = [this, &presenceJsonToCheck](const ttv::social::Friend& friendInList) -> bool {
      ttv::social::PresenceUserAvailability availability;
      const json::Value availabilityJson = presenceJsonToCheck["availability"];
      EXPECT_TRUE(availabilityJson.isString());
      EXPECT_TRUE(ttv::social::ParsePresenceUserAvailability(availabilityJson.asString(), availability));

      Timestamp lastUpdate;
      EXPECT_TRUE(ParseTimestamp(presenceJsonToCheck["updated_at"], lastUpdate));
      return (availability == friendInList.presenceStatus.availability) &&
             (lastUpdate == friendInList.presenceStatus.lastUpdate) &&
             ComparePresenceActivity(presenceJsonToCheck["activity"], friendInList.presenceStatus.activity.get());
    };

    for (std::size_t index = 0; index < 3; index++) {
      presenceJsonToCheck = mFriendPresenceEntries[index];
      ASSERT_FALSE(std::find_if(friends.begin(), friends.end(), compareEntries) == friends.end());
    }

    fetchComplete = false;

    std::vector<ttv::social::FriendRequest> requests;
    ec = mFriendList->FetchFriendRequests(
      [&fetchComplete, &requests](TTV_ErrorCode ec, const std::vector<ttv::social::FriendRequest>& fetchedRequests) {
        EXPECT_TRUE(TTV_SUCCEEDED(ec));
        requests = fetchedRequests;
        fetchComplete = true;
      });

    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    WaitUntilResult(1000, [&fetchComplete]() { return fetchComplete; });

    ASSERT_EQ(requests.size(), 2);

    ttv::social::SocialFriendRequestsTask::FriendRequestEntry friendRequestToCheck;
    auto compareFriendRequest = [&friendRequestToCheck](const ttv::social::FriendRequest& request) {
      return (friendRequestToCheck.userInfo == request.userInfo) &&
             (friendRequestToCheck.requestedTimestamp == request.requestTime);
    };

    for (std::size_t index = 0; index < 2; index++) {
      friendRequestToCheck = mFriendRequests[index];
      ASSERT_FALSE(std::find_if(requests.begin(), requests.end(), compareFriendRequest) == requests.end());
    }

    ASSERT_TRUE(WaitUntilResult(1000, [this]() {
      return mPubSubTestUtility.IsSubscribedToTopic(
               "presence." + std::to_string(mUserTestUtility.DefaultUserInfo().userId)) &&
             mPubSubTestUtility.IsSubscribedToTopic(
               "friendship." + std::to_string(mUserTestUtility.DefaultUserInfo().userId));
    }));
  }

  bool ComparePresenceActivity(const json::Value& json, const ttv::social::PresenceActivity* activity) {
    const auto& jType = json["type"];
    if (activity == nullptr) {
      return json.isNull();
    }

    auto type = jType.asString();
    if (type == "broadcasting") {
      if (activity->GetType() != ttv::social::PresenceActivity::Type::Broadcasting) {
        return false;
      }

      const ttv::social::BroadcastingActivity* broadcastingActivity =
        static_cast<const ttv::social::BroadcastingActivity*>(activity);
      ChannelId channelId;
      if (!ParseChannelId(json["channel_id"], channelId)) {
        return false;
      }
      return (channelId == broadcastingActivity->channelId) &&
             (json["channel_login"] == broadcastingActivity->channelLogin) &&
             (json["channel_display_name"] == broadcastingActivity->channelDisplayName);
    } else if (type == "watching") {
      if (activity->GetType() != ttv::social::PresenceActivity::Type::Watching) {
        return false;
      }

      const ttv::social::WatchingActivity* watchingActivity =
        static_cast<const ttv::social::WatchingActivity*>(activity);

      ChannelId channelId;
      if (!ParseChannelId(json["channel_id"], channelId)) {
        return false;
      }
      ChannelId hostedChannelId;
      if (!ParseChannelId(json["hosted_channel_id"], hostedChannelId)) {
        hostedChannelId = 0;
      }

      return (channelId == watchingActivity->channelId) && (hostedChannelId == watchingActivity->hostedChannelId) &&
             (json["channel_login"] == watchingActivity->channelLogin) &&
             (json["channel_display_name"] == watchingActivity->channelDisplayName) &&
             (json["hosted_channel_login"] == watchingActivity->hostedChannelLogin) &&
             (json["hosted_channel_display_name"] == watchingActivity->hostedChannelDisplayName);
    } else if (type == "playing") {
      if (activity->GetType() != ttv::social::PresenceActivity::Type::Playing) {
        return false;
      }

      const ttv::social::PlayingActivity* playingActivity = static_cast<const ttv::social::PlayingActivity*>(activity);

      GameId gameId;
      if (!ParseGameId(json["game_id"], gameId)) {
        return false;
      }

      return (gameId == playingActivity->gameId) && (json["game"] == playingActivity->gameName) &&
             (json["game_display_context"] == playingActivity->gameDisplayContext);
    } else {
      return (activity->GetType() == ttv::social::PresenceActivity::Type::Unknown);
    }
  }

  bool CompareJsonToCString(const json::Value& json, const char* cString) {
    if (json.isNull()) {
      return (cString == nullptr);
    }
    return json.isString() && (json.asString() == cString);
  }

 protected:
  std::shared_ptr<ttv::social::FriendList> mFriendList;
  std::shared_ptr<ttv::social::FriendListListenerProxy> mFriendListListenerProxy;
  std::vector<json::Value> mFriendPresenceEntries;
  std::vector<ttv::social::SocialFriendRequestsTask::FriendRequestEntry> mFriendRequests;
  std::vector<UserInfo> mRecommendedFriends;
  ttv::test::SampleFriendData mSampleData;
};
}  // namespace

TEST_F(FriendListTest, FetchRecommendedFriends) {
  auto recommendedFriendInfo = mSampleData.GetFriendInfo(5);
  mRecommendedFriends.push_back(recommendedFriendInfo);

  json::FastWriter writer;
  mHttpRequest
    ->AddResponse("https://api.twitch.tv/kraken/users/" + std::to_string(mUserTestUtility.DefaultUserInfo().userId) +
                  "/friends/recommendations")
    .SetResponseBody(writer.write(RecommendedFriendFetchResponse()))
    .Done();

  bool recommendedFriendFetchComplete = false;
  std::vector<UserInfo> recommendedFriends;
  TTV_ErrorCode ec =
    mFriendList->FetchRecommendedFriends([&recommendedFriendFetchComplete, &recommendedFriends](
                                           TTV_ErrorCode ec, const std::vector<UserInfo>& fetchedRecommendations) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      recommendedFriends = fetchedRecommendations;
      recommendedFriendFetchComplete = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [&recommendedFriendFetchComplete]() { return recommendedFriendFetchComplete; }, GetDefaultUpdateFunc()));

  ASSERT_EQ(recommendedFriends.size(), 1);

  ASSERT_FALSE(std::find_if(recommendedFriends.begin(), recommendedFriends.end(),
                 [&recommendedFriendInfo](const UserInfo& userInfo) { return recommendedFriendInfo == userInfo; }) ==
               recommendedFriends.end());
}

TEST_F(FriendListTest, FriendActivityChange) {
  // The idle friend comes online and changes activity.
  auto friendInfo = mSampleData.GetFriendInfo(2);
  auto newActivityJson = mSampleData.GetActivityJson(2);
  auto newPresenceJson = mSampleData.GeneratePresenceJson(friendInfo, mSampleData.GenerateRandomTimestamp(),
    mSampleData.GenerateRandomTimestamp(), 1, ttv::social::PresenceUserAvailability::Online, newActivityJson);

  bool friendInfoChanged = false;
  mFriendListListenerProxy->mOnFriendInfoChangedFunc = [this, &friendInfoChanged, &friendInfo, &newActivityJson](
                                                         ttv::social::FriendList* /*source*/,
                                                         const std::vector<ttv::social::Friend>& list) {
    ASSERT_EQ(list.size(), 1);
    ASSERT_EQ(friendInfo, list[0].userInfo);
    ASSERT_TRUE(ComparePresenceActivity(newActivityJson, list[0].presenceStatus.activity.get()));
    friendInfoChanged = true;
  };

  json::Value pubSubMessage;
  pubSubMessage["type"] = "presence";
  pubSubMessage["data"] = newPresenceJson;

  json::FastWriter writer;
  mPubSubTestUtility.PushPubSubMessage(
    "presence." + std::to_string(mUserTestUtility.DefaultUserInfo().userId), writer.write(pubSubMessage));

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&friendInfoChanged]() { return friendInfoChanged; }, GetDefaultUpdateFunc()));

  ASSERT_TRUE(TTV_SUCCEEDED(mFriendList->FetchFriendList([this, &friendInfo, &newActivityJson](TTV_ErrorCode /*ec*/,
                                                           const std::vector<ttv::social::Friend>& fetchedFriends) {
    ASSERT_EQ(fetchedFriends.size(), 3);
    auto changedFriend = std::find_if(fetchedFriends.begin(), fetchedFriends.end(),
      [&friendInfo](const ttv::social::Friend& fetchedFriend) { return friendInfo == fetchedFriend.userInfo; });

    ASSERT_NE(changedFriend, fetchedFriends.end());
    ASSERT_TRUE(ComparePresenceActivity(newActivityJson, changedFriend->presenceStatus.activity.get()));
  })));
}

TEST_F(FriendListTest, TestShutdownWithPendingFriendRequests) {
  json::FastWriter writer;
  mFriendList->FetchFriendList(nullptr);

  // The test tear down will wait for the shut down of all components,
  // including the friend list.
}
