
/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2015-2016 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "fixtures/sdkbasetest.h"
#include "pubsubtestutility.h"
#include "socialtestutilities.h"
#include "testutilities.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/social/internal/presence.h"
#include "usertestutility.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;

namespace {
const UserId kTestUserId = 9001;

class SocialPresenceTest : public SdkBaseTest {
 protected:
  virtual void SetUpStubs() {
    SdkBaseTest::SetUpStubs();

    mHttpRequest->AddResponse("https://api.twitch.tv/v5/users/" + std::to_string(kTestUserId) + "/status/settings")
      .SetResponseBodyFromFile("social/socialpresencesettingstask.json")
      .Done();
  }

  virtual void SetUpComponents() {
    SdkBaseTest::SetUpComponents();

    auto userRepository = CreateUserRepository();
    InitializeComponent(userRepository);

    UserInfo userInfo;
    userInfo.userName = "test_user_name";
    userInfo.displayName = "test_display_name";
    userInfo.userId = kTestUserId;

    auto user = GetUserRepository()->RegisterUser(kTestUserId);
    user->SetOAuthToken(std::make_shared<OAuthToken>("auth_token"));
    user->SetUserInfo(userInfo);

    mPubSubTestUtility.SetUpComponents(CreateTaskRunner(), user);
    InitializeComponent(mPubSubTestUtility.GetPubSubClient());

    mPresenceListener = std::make_shared<ttv::social::PresenceListenerProxy>();

    mPresence = std::make_shared<ttv::social::Presence>(user);
    mPresence->SetAutomaticPresencePostingEnabled(false);
    mPresence->AddListener(mPresenceListener);
    mPresence->SetTaskRunner(CreateTaskRunner());
    AddComponent(mPresence);
  }

  virtual void TearDownComponents() {
    mPubSubTestUtility.TearDownComponents();

    SdkBaseTest::TearDownComponents();
  }

  virtual void TearDownStubs() { SdkBaseTest::TearDownStubs(); }

  std::string RequestUrl(UserId userId) {
    std::stringstream urlStream;
    urlStream << "https://api.twitch.tv/v5/users/" << userId << "/status";
    return urlStream.str();
  }

 protected:
  std::shared_ptr<ttv::social::Presence> mPresence;
  std::shared_ptr<ttv::social::PresenceListenerProxy> mPresenceListener;
};
}  // namespace

TEST_F(SocialPresenceTest, SocialPresence_SubscribeAndReceiveMessage) {
  auto presenceTopic = "presence." + std::to_string(kTestUserId);

  bool settingsChanged = false;

  ttv::social::PresenceSettings settings;
  mPresenceListener->mOnSettingsChangedFunc = [&settingsChanged, &settings](ttv::social::Presence* /*presence*/,
                                                const ttv::social::PresenceSettings& fetchedSettings) {
    settingsChanged = true;
    settings = fetchedSettings;
  };

  InitializeComponent(mPresence);

  bool settingsFetchCompleted = false;
  TTV_ErrorCode ec = mPresence->FetchSettings(
    [&settingsFetchCompleted, &settings](TTV_ErrorCode ec, const ttv::social::PresenceSettings& fetchedSettings) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      settingsFetchCompleted = true;
      settings = fetchedSettings;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this, &presenceTopic]() { return mPubSubTestUtility.IsSubscribedToTopic(presenceTopic); },
    GetDefaultUpdateFunc()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [&settingsFetchCompleted]() { return settingsFetchCompleted; }, GetDefaultUpdateFunc()));

  ASSERT_TRUE(settings.shareActivity);

  settingsChanged = false;

  json::FastWriter writer;

  json::Value message;
  json::Value data;
  data["availability_override"] = "";
  data["share_activity"] = false;
  message["type"] = "settings";
  message["data"] = data;

  mPubSubTestUtility.PushPubSubMessage(presenceTopic, writer.write(message));

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&settingsChanged]() { return settingsChanged; }, GetDefaultUpdateFunc()));

  ASSERT_FALSE(settings.shareActivity);
}

TEST_F(SocialPresenceTest, SocialPresence_PostPresenceSuccess) {
  constexpr char response[] = R"(
{
  "data": {
    "setSessionStatus": {
      "setAgainInSeconds": 60
    }
  }
}
)";

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBody(response)
    .AddRequestRegex(std::regex("SetSessionStatus"))
    .Done();

  auto expectedAvailabilityString = "online";
  std::function<void(const json::Value& activityJson)> activityChecker = [](const json::Value& activityJson) {
    ASSERT_TRUE(activityJson.isNull());
  };

  InitializeComponent(mPresence);
  bool presencePosted = false;
  auto postPresenceCallback = [&presencePosted](TTV_ErrorCode ec) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    presencePosted = true;
  };
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence(postPresenceCallback)));

  auto presenceCheck = [&presencePosted]() { return presencePosted; };

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, presenceCheck, GetDefaultUpdateFunc()));

  presencePosted = false;

  activityChecker = [](const json::Value& activitiesJson) {
    ASSERT_FALSE(activitiesJson.isNull());
    ASSERT_TRUE(activitiesJson.isArray());
    ASSERT_EQ(activitiesJson.size(), 1);

    const auto& activityJson = activitiesJson[0];
    const auto& type = activityJson["type"];
    ASSERT_TRUE(type.isString());
    ASSERT_EQ(type.asString(), "watching");

    ChannelId channelId;
    ASSERT_TRUE(ParseChannelId(activityJson["channel_id"], channelId));
    ASSERT_EQ(channelId, 1337);
  };

  uint32_t activityToken;
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->AddWatchingActivity(1337, activityToken)));
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence(postPresenceCallback)));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, presenceCheck, GetDefaultUpdateFunc()));

  presencePosted = false;
  activityChecker = [](const json::Value& activityJson) { ASSERT_TRUE(activityJson.isNull()); };
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->RemoveActivity(activityToken)));

  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence(postPresenceCallback)));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, presenceCheck, GetDefaultUpdateFunc()));
  presencePosted = false;
  expectedAvailabilityString = "idle";
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->SetSessionAvailability(ttv::social::PresenceSessionAvailability::Idle)));
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence(postPresenceCallback)));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, presenceCheck, GetDefaultUpdateFunc()));

  presencePosted = false;
  expectedAvailabilityString = "offline";
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->SetSessionAvailability(ttv::social::PresenceSessionAvailability::Offline)));
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence(postPresenceCallback)));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, presenceCheck, GetDefaultUpdateFunc()));
}

TEST_F(SocialPresenceTest, SocialPresence_PostPresence404Failure) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(404)
    .AddRequestRegex(std::regex("SetSessionStatus"))
    .Done();

  bool presencePosted = false;
  InitializeComponent(mPresence);
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence([&presencePosted](TTV_ErrorCode ec) {
    ASSERT_EQ(ec, TTV_EC_API_REQUEST_FAILED);
    presencePosted = true;
  })));

  WaitUntilResultWithPollTask(1000, [&presencePosted]() { return presencePosted; }, GetDefaultUpdateFunc());
}

TEST_F(SocialPresenceTest, SocialPresence_PostPresence401Failure) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(401)
    .AddRequestRegex(std::regex("SetSessionStatus"))
    .Done();

  bool presencePosted = false;
  InitializeComponent(mPresence);
  ASSERT_TRUE(TTV_SUCCEEDED(mPresence->PostPresence([&presencePosted](TTV_ErrorCode ec) {
    ASSERT_EQ(ec, TTV_EC_AUTHENTICATION);
    presencePosted = true;
  })));

  WaitUntilResultWithPollTask(1000, [&presencePosted]() { return presencePosted; }, GetDefaultUpdateFunc());
}
